import React, { ReactNode, useRef } from 'react';
import { DotChip, DotTooltip } from '@digital-ai/dot-components';
import classNames from 'classnames';
import { useChipGroup } from './hooks/useChipGroup.hook';
import './chip-group.component.less';

export interface ChipGroupProps {
    chipCharactersLimit?: number;
    className?: string;
    labels: string[];
    onChipDelete?: (label: string) => void;
}

export interface ExceedingWidthChip {
    index: number;
    label: string;
}

export const ChipGroup = ({ chipCharactersLimit, className, labels }: ChipGroupProps) => {
    const sortedLabels = [...labels].sort((a, b) => a.length - b.length);
    const containerRef = useRef<HTMLDivElement | null>(null);
    const exceedingChipsCountRef = useRef<HTMLSpanElement | null>(null);
    const elementsRef = useRef<HTMLSpanElement[]>([]);
    let firstItemExceedingWidth = 0;

    const { chipWidths, containerWidth, chipCounterWidth } = useChipGroup(exceedingChipsCountRef, elementsRef, containerRef);

    const exceedingWidthChips: ExceedingWidthChip[] = [];
    let remainingContainerWidth = containerWidth;

    const calculateGapWidth = (index: number, counterWidth: number) => {
        const isLastVisibleItem = index >= chipWidths.length - exceedingWidthChips.length - 1;
        const isChipCounterVisible = counterWidth > 0;
        return isLastVisibleItem && !isChipCounterVisible ? 0 : 8;
    };

    chipWidths.forEach((width, index) => {
        const gapWidth = calculateGapWidth(index, chipCounterWidth);
        const totalTakenWidth = width + gapWidth;

        const isExceedingWidth = totalTakenWidth > remainingContainerWidth - chipCounterWidth;

        if (isExceedingWidth && index === 0) {
            firstItemExceedingWidth = totalTakenWidth + chipCounterWidth - remainingContainerWidth;
        } else if (isExceedingWidth && index > 0) {
            exceedingWidthChips.push({ index, label: sortedLabels[index] });
        } else {
            remainingContainerWidth -= totalTakenWidth;
        }
    });

    const getTooltipWithExceedingChips = (): ReactNode => {
        return (
            <>
                {exceedingWidthChips.map((chip) => (
                    <div key={chip.label}>
                        {chip.label}
                        <br />
                    </div>
                ))}
            </>
        );
    };

    return (
        <div className={classNames('chip-group', className)} ref={containerRef}>
            {sortedLabels.map((label: string, index: number) => {
                return (
                    <span
                        className={exceedingWidthChips.some((exceedingChip) => exceedingChip.index === index) ? 'hidden-chip' : undefined}
                        data-testid="chip-group-span"
                        key={label}
                        ref={(element) => {
                            if (element !== null) {
                                elementsRef.current[index] = element;
                            }
                        }}
                        style={{ width: index === 0 && index in chipWidths ? chipWidths[index] - firstItemExceedingWidth : undefined }}
                    >
                        <DotChip charactersLimit={chipCharactersLimit} size="small">
                            {label}
                        </DotChip>
                    </span>
                );
            })}
            {exceedingWidthChips.length > 0 && (
                <span data-testid="exceeding-width-chip-span" ref={exceedingChipsCountRef}>
                    <DotTooltip title={getTooltipWithExceedingChips()}>
                        <DotChip size="small">{`+${exceedingWidthChips.length.toString()}`}</DotChip>
                    </DotTooltip>
                </span>
            )}
        </div>
    );
};
