import React from 'react';
import ReactDOM from 'react-dom';
import constant from 'lodash/constant';
import { mount, shallow } from 'enzyme/build';
import { Suggestions } from './suggestions.component';
import { DEFAULT_LABEL_FIELD } from './xlr-tags-constants';

const defaults = {
    query: 'ang',
    suggestions: [
        { id: 'Banana', text: 'Banana' },
        { id: 'Mango', text: 'Mango' },
        { id: 'Pear', text: 'Pear' },
        { id: 'Apricot', text: 'Apricot' },
    ],
    labelField: DEFAULT_LABEL_FIELD,
    selectedIndex: 1,
    isFocused: true,
    isShowFromBottom: false,
    handleClick: jest.fn(),
    handleHover: jest.fn(),
    classNames: { suggestions: 'foo', activeSuggestion: 'active' },
};

function mockItem(overrides) {
    const props = Object.assign({}, defaults, overrides);
    return <Suggestions {...props} />;
}

describe('Suggestions component', () => {
    test('should render with expected props', () => {
        const suggestionComponent = shallow(mockItem());
        const expectedProps = {
            minQueryLength: 2,
            ...defaults,
        };

        expect(suggestionComponent.instance().props).toEqual(expectedProps);
    });

    test('shows the classname properly', () => {
        const suggestionComponent = shallow(mockItem());
        expect(suggestionComponent.find('.foo').length).toBe(1);
    });

    test('renders all suggestions properly', () => {
        const suggestionComponent = shallow(mockItem());
        expect(suggestionComponent.find('li').length).toBe(4);
    });

    test('selects the correct suggestion', () => {
        const suggestionComponent = mount(mockItem());
        expect(suggestionComponent.find('li.active').length).toBe(1);
        expect(suggestionComponent.find('li.active').text()).toBe('Mango');
    });

    test('should not render suggestion with less than minQueryLength', () => {
        const suggestionComponent = shallow(
            mockItem({
                minQueryLength: 2,
                query: 'q',
            }),
        );
        expect(suggestionComponent.find('.foo').length).toBe(0);
        expect(suggestionComponent.find('li').length).toBe(0);
    });

    test('should be able to override suggestion render', () => {
        const suggestionComponent = shallow(
            mockItem({
                minQueryLength: 2,
                query: 'ignore_query',
                shouldRenderSuggestions: (q) => q !== 'ignore_query',
            }),
        );
        expect(suggestionComponent.find('.foo').length).toBe(0);
        expect(suggestionComponent.find('li').length).toBe(0);
    });

    test('should escape html characters in query', () => {
        const suggestions = [{ id: 'script', text: '<script>alert()</script>' }];
        const suggestionComponent = shallow(
            mockItem({
                query: '<script>alert()</script>',
                suggestions,
            }),
        );
        expect(suggestionComponent.html()).toBe('<div class="foo"><ul> <li class=""><span>&lt;script&gt;alert()&lt;/script&gt;</span></li> </ul></div>');
    });

    test('should mark highlighted suggestions correctly', () => {
        const suggestionComponent = shallow(mockItem());
        expect(suggestionComponent.find('li.active').find('span').html()).toBe('<span>Mango</span>');
    });

    test('should not wastefully re-render if the list of suggestions have not changed', () => {
        const suggestions = [
            { id: 'queue', text: 'queue' },
            { id: 'quiz', text: 'quiz' },
            { id: 'quantify', text: 'quantify' },
        ];
        const suggestionComponent = mount(
            mockItem({
                minQueryLength: 2,
                query: 'q',
                suggestions,
            }),
        );

        Suggestions.prototype.componentDidUpdate = jest.fn();
        suggestionComponent.setProps({ suggestions });
        expect(Suggestions.prototype.componentDidUpdate).toHaveBeenCalledTimes(0);
    });

    test('should re-render if there is an active query', () => {
        const suggestions = [
            { id: 'queue', text: 'queue' },
            { id: 'quiz', text: 'quiz' },
            { id: 'quantify', text: 'quantify' },
        ];
        const suggestionComponent = mount(
            mockItem({
                minQueryLength: 2,
                query: 'qu',
                suggestions,
            }),
        );
        Suggestions.prototype.componentDidUpdate = jest.fn();
        suggestionComponent.setProps({ suggestions });
        expect(Suggestions.prototype.componentDidUpdate).toHaveBeenCalledTimes(1);
    });

    test('should re-render if minQueryLength is set to 0', () => {
        const suggestions = [
            { id: 'queue', text: 'queue' },
            { id: 'quiz', text: 'quiz' },
            { id: 'quantify', text: 'quantify' },
        ];
        const suggestionComponent = mount(
            mockItem({
                minQueryLength: 0,
                query: '',
                suggestions,
            }),
        );
        Suggestions.prototype.componentDidUpdate = jest.fn();
        suggestionComponent.setProps({ suggestions });
        expect(Suggestions.prototype.componentDidUpdate).toHaveBeenCalledTimes(1);
    });

    test('should re-render if the provided "shouldRenderSuggestions" prop returns true', () => {
        const suggestions = [
            { id: 'queue', text: 'queue' },
            { id: 'quiz', text: 'quiz' },
            { id: 'quantify', text: 'quantify' },
        ];
        const suggestionComponent = mount(
            mockItem({
                shouldRenderSuggestions: constant(true),
                suggestions,
            }),
        );
        Suggestions.prototype.componentDidUpdate = jest.fn();
        suggestionComponent.setProps({ suggestions });
        expect(Suggestions.prototype.componentDidUpdate).toHaveBeenCalledTimes(1);
    });

    test('should re-render when the query reaches minQueryLength', () => {
        const suggestions = [
            { id: 'queue', text: 'queue' },
            { id: 'quiz', text: 'quiz' },
            { id: 'quantify', text: 'quantify' },
        ];
        const div = document.createElement('div');
        const component = mockItem({
            minQueryLength: 2,
            query: '',
            suggestions,
        });

        /* eslint-disable react/no-render-return-value */
        ReactDOM.render(component, div);
        /* eslint-enable react/no-render-return-value */

        // re-render with updated query prop
        Suggestions.prototype.componentDidUpdate = jest.fn();

        /* eslint-disable react/no-render-return-value */
        ReactDOM.render(
            mockItem({
                minQueryLength: 2,
                query: 'qu',
                suggestions,
            }),
            div,
        );
        /* eslint-enable react/no-render-return-value */

        expect(Suggestions.prototype.componentDidUpdate).toHaveBeenCalledTimes(1);
    });

    test('should render custom suggestions when renderSuggestion prop is provided', () => {
        /* eslint-disable react/display-name,  react/prop-types */
        const suggestionComponent = shallow(
            mockItem({
                renderSuggestion: ({ text }) => (
                    <div className="bar">
                        <i />
                        {text}
                    </div>
                ),
            }),
        );
        /* eslint-enable react/display-name, react/prop-types */

        expect(suggestionComponent.find('.bar').length).toBe(4);
    });

    test('should trigger the click handler on touchStart', () => {
        const spy = jest.fn();
        const suggestionComponent = mount(mockItem({ handleClick: spy }));
        suggestionComponent.find('li').first().simulate('touchStart');
        expect(spy).toHaveBeenCalledTimes(1);
    });
});
