import React from 'react';
import { act } from 'react-dom/test-utils';
import { DotTooltip } from '@digital-ai/dot-components';
import constant from 'lodash/constant';
import { TextAreaMarkdown, TextareaMarkdownProps } from './textarea-markdown.component';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../tests/unit/testing-utils';

type ActionTypeId = 'save-button' | 'cancel-button' | 'edit-button' | 'delete-button';

describe('TextAreaMarkdown Component', () => {
    const onTextChange = jest.fn();
    let wrapper: ReactWrapper;

    const defaultProps: TextareaMarkdownProps = {
        inputId: 'markdown-input',
        name: 'markdown',
        onTextChange,
        rows: 5,
        text: 'my **markdown** text',
    };

    const mount = async (props: TextareaMarkdownProps = defaultProps) => {
        await act(async () => {
            wrapper = mountWithStoreAndTheme(<TextAreaMarkdown {...props} />);
        });
    };

    const markdownSwitcherSelector = '.markdown-switcher';
    const markdownViewerSelector = `${markdownSwitcherSelector} .markdown-viewer`;
    const markdownWrapperSelector = `${markdownViewerSelector} .markdown-wrapper`;
    const getTooltip = () => wrapper.find(markdownViewerSelector).findWhere((node) => node.is(DotTooltip) && node.props().title === 'Double-click to edit');
    const selectAction = (testId: ActionTypeId) => `${markdownSwitcherSelector} button[data-testid="${testId}"]`;
    const textareaSelector = `${markdownSwitcherSelector} textarea#markdown-input`;
    const isInEditMode = () => wrapper.find(textareaSelector).exists();
    const getMarkdownSwitcher = () => wrapper.find(markdownSwitcherSelector);
    const getHtmlText = () => getMarkDownWrapperSelector().html();
    const getMarkDownWrapperSelector = () => wrapper.find(markdownWrapperSelector);
    const getPlaceholderWrapper = () => wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
    const getSaveButton = () => wrapper.find(selectAction('save-button'));
    const setMarkdownText = (value: string) => wrapper.find(textareaSelector).simulate('change', { target: { value } });
    const saveChanges = async () => {
        await act(async () => {
            getSaveButton().simulate('click');
        });
        wrapper.update();
    };
    const checkInitialState = () => {
        expect(isInEditMode()).toBeFalsy();
        expect(getHtmlText()).toContain('my <strong>markdown</strong> text');
    };

    const switchToEditMode = () => {
        wrapper.find(selectAction('edit-button')).simulate('click');
        expect(isInEditMode()).toBeTruthy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeFalsy();
    };

    const switchToViewMode = () => {
        wrapper.find(selectAction('cancel-button')).simulate('click');
        expect(isInEditMode()).toBeFalsy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeTruthy();
    };

    const checkEmptyInitialState = () => {
        expect(isInEditMode()).toBeFalsy();
        expect(wrapper.find(markdownWrapperSelector).exists()).toBeFalsy();
    };

    const checkActionButtons = (shouldDeleteBtnExist = false) => {
        const actions = wrapper.find(`${markdownViewerSelector} .markdown-viewer-actions`);
        expect(actions.exists()).toBeTruthy();
        expect(actions.find('[data-testid="delete-button"]').exists()).toBe(shouldDeleteBtnExist);
        expect(actions.find('[data-testid="edit-button"]').exists()).toBeTruthy();
    };
    beforeEach(() => {
        jest.resetAllMocks();
    });

    it('should render markdown by default', async () => {
        await mount();
        checkInitialState();
        checkActionButtons();
    });

    it('should render with correct class name', async () => {
        const className = 'test-markdown';
        await mount({ ...defaultProps, className });
        const switcher = getMarkdownSwitcher();
        expect(switcher).toHaveClassName(className);
    });

    it('should render tooltip with disabled portal', async () => {
        await mount();
        const tooltip = getTooltip();
        expect(tooltip.props().disablePortal).toBe(true);
    });

    it('should render markdown with mentions', async () => {
        const users = [
            { username: 'admin', fullName: 'Release Administrator' },
            { username: 'bob', fullName: 'Bob Builder' },
            { username: 'suzy', fullName: 'Suzy' },
            { username: 'ann', fullName: '' },
            { username: 'Itchy' },
            { username: 'Scratchy' },
        ];
        const text = `## My **Manual task**. @admin can you check?  
        
@ann assign @itchy and @unknown to task  
        
This should be solved by@Scratchy or by **@suzy**  
        
**Note: @bob is PO**`;
        await mount({ ...defaultProps, users, enableMentions: true, text });
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        const html = getHtmlText();

        expect(html).toContain('<h2>My <strong>Manual task</strong>');
        expect(html).toContain('<span class="mention" title="admin">@Release Administrator</span>');
        expect(html).toContain('<span class="mention" title="ann">@ann</span> assign <strong>@itchy</strong> and <strong>@unknown</strong> to task</p>');
        expect(html).toContain('This should be solved by@Scratchy or by <span class="mention" title="suzy">@Suzy</span>');
        expect(html).toContain('<strong>Note: <span class="mention" title="bob">@Bob Builder</span> is PO</strong>');
    });

    it('should render markdown with email mentions for OIDC', async () => {
        const users = [{ username: 'bruno.devic@digital.ai', fullName: 'Release Administrator' }];
        const text = `Hey @bruno.devic@digital.ai can you check?`;

        await mount({ ...defaultProps, users, enableMentions: true, text });
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        const html = getHtmlText();

        expect(html).toContain('Hey <span class="mention" title="bruno.devic@digital.ai">@Release Administrator</span> can you check?');
    });

    it('should switch to edit mode', async () => {
        await mount();
        // initial markdown mode
        checkInitialState();
        // edit mode
        switchToEditMode();
        // markdown mode
        switchToViewMode();
        expect(onTextChange).not.toHaveBeenCalled();
    });

    it('should render edit mode with mentions', async () => {
        await mount({ ...defaultProps, enableMentions: true });
        checkInitialState();
        switchToEditMode();
        expect(wrapper.find('OptionSelectorInput').exists()).toBeTruthy();
    });

    it('should render bold links when mentions is enabled', async () => {
        await mount({ ...defaultProps, enableMentions: true, text: '**[Google](https://www.google.com)**' });
        expect(getHtmlText()).toContain('<strong><a href="https://www.google.com" target="_blank">Google</a></strong>');
    });

    it('should save and cancel changes', async () => {
        await mount();
        // original text
        checkInitialState();

        // do change
        switchToEditMode();
        setMarkdownText('my ```fancy code```');
        // cancel change
        switchToViewMode();
        // original state
        checkInitialState();
        expect(getHtmlText()).not.toContain('fancy code');
        // do change
        switchToEditMode();
        setMarkdownText('my ```fancy code```');
        // save change
        await saveChanges();
        expect(onTextChange).toHaveBeenCalledWith('my ```fancy code```');
        expect(getHtmlText()).toContain('my <code>fancy code</code>');
    });

    it('should render disabled Save button when empty space is entered and value is required', async () => {
        await mount();
        // original text
        checkInitialState();

        // do change
        switchToEditMode();
        setMarkdownText('    ');
        expect(getSaveButton()).toBeDisabled();
    });

    it('should render disabled Save button when value is required but empty', async () => {
        await mount();
        checkInitialState();
        switchToEditMode();
        setMarkdownText('');
        expect(getSaveButton()).toBeDisabled();
    });

    it('should render delete button when onDelete function is defined', async () => {
        await mount({ ...defaultProps, onDelete: jest.fn() });
        checkInitialState();
        checkActionButtons(true);
    });

    it('should not delete original text when clicking on delete', async () => {
        await mount({ ...defaultProps, onDelete: jest.fn() });
        expect(getHtmlText()).toContain('my <strong>markdown</strong> text');
        wrapper.find('button[data-testid="delete-button"]').simulate('click');
        expect(getHtmlText()).toContain('my <strong>markdown</strong> text');
    });

    it('should render empty placeholder when value is empty', async () => {
        await mount({ ...defaultProps, text: '' });
        checkEmptyInitialState();
        checkActionButtons();

        const placeholderWrapper = wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('');
    });

    it('should render placeholder with text when value is empty', async () => {
        await mount({ ...defaultProps, text: '', placeholder: 'Add text...' });
        checkEmptyInitialState();
        checkActionButtons();

        const placeholderWrapper = getPlaceholderWrapper();
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('Add text...');
    });

    it('should replace variable and wrap it into into variable span when enableVariables is set to true', async () => {
        const variables = { '${var}': 'variableValue' };
        const text = 'my ${var} text with variables';
        await mount({ ...defaultProps, enableVariables: true, releaseVariables: variables, text });
        expect(getHtmlText()).toContain('my <span class="variable">variableValue</span> text with variables');
    });

    it('should NOT replace variable when enableVariables is set to false', async () => {
        const variables = { '${var}': 'variableValue' };
        const text = 'my ${var} text with variables';
        await mount({ ...defaultProps, enableVariables: false, releaseVariables: variables, text });
        expect(getHtmlText()).toContain(text);
    });

    it('should show placeholder when empty value is set but value is not required', async () => {
        await mount({ ...defaultProps, text: 'this will be removed', isValueRequired: false, placeholder: 'this my placeholder' });
        switchToEditMode();
        setMarkdownText('');
        // save change
        await saveChanges();

        expect(getPlaceholderWrapper()).toExist();
        expect(onTextChange).toHaveBeenCalledWith('');
    });

    it('should render links inside Release', async () => {
        const link = `${document.location.origin}/#/releases`;
        const text = `[content](${link})`;
        await mount({ ...defaultProps, text });
        expect(getHtmlText()).toContain('<a href="http://localhost/#/releases">content</a>');
    });

    it('should render links outside Release', async () => {
        const link = `https://www.google.com`;
        const text = `[content](${link})`;
        await mount({ ...defaultProps, text });
        expect(getHtmlText()).toContain('<a href="https://www.google.com" target="_blank">content</a>');
    });

    describe('when clicking on editor', () => {
        const clickOnViewer = (target: HTMLElement) => {
            wrapper.find(markdownViewerSelector).simulate('mouseup', { target });
        };
        const doubleClickOnViewer = (target: HTMLElement) => {
            wrapper.find(markdownViewerSelector).simulate('dblclick', { target });
        };

        it('should not go to edit mode if text is selected', async () => {
            const getSelection = jest.fn();
            getSelection.mockReturnValue({
                toString: constant('my mocked selected text'),
            });
            window.document.getSelection = getSelection;

            await mount(defaultProps);
            clickOnViewer(document.createElement('div'));
            expect(isInEditMode()).toBe(false);
        });

        it('should not go to edit mode if link is clicked', async () => {
            await mount(defaultProps);
            const fakeLink = document.createElement('a');
            fakeLink.setAttribute('href', 'www.9gag.com');
            clickOnViewer(fakeLink);
            expect(isInEditMode()).toBe(false);
        });

        it('should not go to edit mode if delete button is clicked', async () => {
            await mount(defaultProps);
            const fakeDotIcon = document.createElement('i');
            fakeDotIcon.setAttribute('class', 'dot-i');
            clickOnViewer(fakeDotIcon);
            expect(isInEditMode()).toBe(false);
        });

        it('should not go to edit mode if component is in readOnly', async () => {
            mount({ ...defaultProps, isReadOnly: true });
            clickOnViewer(document.createElement('div'));
            expect(isInEditMode()).toBe(false);
        });

        it('should go to edit mode if viewer is clicked', async () => {
            await mount(defaultProps);
            doubleClickOnViewer(document.createElement('div'));
            expect(isInEditMode()).toBe(true);
        });
    });
});
