import React, { Component } from 'react';
import flattenDeep from 'lodash/flattenDeep';
import PropTypes from 'prop-types';
import { getAllStepperTasks, getAllSubTasks, isTaskCompleted, isTaskCompletedInAdvance } from '../../../../features/tasks/helper/task.helper';
import './release-progress.component.less';

export const calculatePercentageForReleaseOrWorkflow = (release, isStepperView) => {
    return isStepperView ? calculatePercentageForStepper(release) : calculatePercentage(release.extensions?.progress);
};

export const calculatePercentageForStepper = (release) => {
    const tasks = getAllStepperTasks(release);
    const allTasks = flattenDeep(tasks.map(getAllSubTasks));
    const totalTasks = allTasks.length;
    const totalRemainingTasks = allTasks.filter((task) => !isTaskCompleted(task) && !isTaskCompletedInAdvance(task)).length;
    return calculateCompletionPercentage(totalTasks, totalRemainingTasks);
};

export const calculatePercentage = (progress) => {
    if (!progress) {
        return 0;
    }
    const totalTasks = progress.totalTasks;
    const totalRemainingTasks = progress.totalRemainingTasks;
    return calculateCompletionPercentage(totalTasks, totalRemainingTasks);
};

const calculateCompletionPercentage = (totalTasks, totalRemainingTasks) => {
    const tasksFinished = totalTasks - totalRemainingTasks;
    return tasksFinished === 0 ? 0 : Math.floor((tasksFinished * 100) / totalTasks);
};

export class ReleaseProgressComponent extends Component {
    static propTypes = {
        completedPercentage: PropTypes.number,
        progress: PropTypes.shape({
            totalTasks: PropTypes.number.isRequired,
            totalRemainingTasks: PropTypes.number.isRequired,
        }),
    };

    render() {
        const { completedPercentage, progress } = this.props;

        let percentage = 0;
        if (completedPercentage) {
            percentage = completedPercentage;
        } else if (progress) {
            percentage = calculatePercentage(progress);
        }

        const barWidthStyle = {
            width: percentage + '%',
        };

        return (
            <div className="release-progress-bar-view">
                <div className="percent-number">{percentage}% Completed</div>
                <div className="progress-bar-container">
                    <div className="progress-bar-line" style={barWidthStyle} />
                </div>
            </div>
        );
    }
}
