package com.xebialabs.xltest.cucumber;

import com.jayway.jsonpath.InvalidPathException;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

import static com.jayway.jsonpath.JsonPath.read;
import static com.xebialabs.xltest.importers.ImporterUtil.sendEvent;

public class CucumberReportJsonImporter implements Importable {
    private static final Logger LOG = LoggerFactory.getLogger(CucumberReportJsonImporter.class);

    private final OverthereFile cucumberReportJsonFile;

    public CucumberReportJsonImporter(OverthereFile cucumberReportJsonFile) {
        this.cucumberReportJsonFile = cucumberReportJsonFile;
    }

    @Override
    public void doImport(UUID testRunId, EventHandler eventHandler) throws ImportException {
        long started = cucumberReportJsonFile.lastModified();
        sendEvent(eventHandler, testRunId, Event.IMPORT_STARTED, Event.props(
                "lastModified", started,
                "fileName", cucumberReportJsonFile.getPath()
        ));

        int totalDuration = 0;

        try {
            List<Object> features;
            try (InputStream fis = cucumberReportJsonFile.getInputStream()) {
                features = (List<Object>) JsonProviderFactory.createProvider().parse(fis);
            } catch (IOException | ClassCastException e) {
                throw new NothingToImportException("Could not process file " + cucumberReportJsonFile, e);
            }

            try {
                for (Object feature : features) {
                    int duration = tellAboutFeature((Map<String, Object>) feature, testRunId, eventHandler);
                    totalDuration = totalDuration + duration;
                }
            } catch (ClassCastException e) {
                throw new ImportFailedException("Could not process file " + cucumberReportJsonFile, e);
            }

        } finally {
            sendEvent(eventHandler, testRunId, Event.IMPORT_FINISHED, Event.props(
                    "duration", totalDuration
            ));
        }
        LOG.info("Generated start / finish event for file: " + cucumberReportJsonFile.getPath());
    }

    private int tellAboutFeature(Map<String, Object> feature, UUID testRunId, EventHandler eventHandler) throws ImportException {
        String featureId = read(feature, "id");
        String featureName = read(feature, "name");
        List<String> featureTags = readTags(feature);
        List<Map<String, Object>> scenarios;
        try {
            scenarios = read(feature, "elements[?(@.type == 'scenario')]");
        } catch (InvalidPathException e) {
            //Catch InvalidPathException if elements is not available in the json Object
            scenarios = Collections.emptyList();
        }

        int totalDurationInMillis = 0;
        for (Map<String, Object> scenario : scenarios) {
            int duration = scenarioDuration(scenario);
            String scenarioResult = scenarioResult(scenario);
            Event event = new Event(testRunId, Event.FUNCTIONAL_RESULT, Event.props(
                    "result", scenarioResult,
                    "duration", duration,
                    "featureId", featureId,
                    "id", read(scenario, "id"),
                    "name", featureName + Event.TESTNAME_DELIMITER + read(scenario, "name") + Event.TESTNAME_DELIMITER + read(scenario, "line"),
                    // TODO: fix tags defined by test tools
//                    "tags", scenarioTags(scenario, featureTags),
                    "steps", read(scenario, "steps")
            ));
            sendEvent(eventHandler, event);
            totalDurationInMillis = totalDurationInMillis + duration;
        }
        return totalDurationInMillis;
    }

    private List<String> readTags(Map<String, Object> feature) {
        try {
            return read(feature, "$.tags[*].name");
        } catch (InvalidPathException e) {
            return Collections.emptyList();
        }
    }

    private String scenarioResult(Map<String, Object> scenario) {
        List<String> results = read(scenario, "$.steps[*].result.status");
        for (String result : results) {
            if (!"passed".equals(result)) {
                return "FAILED";
            }
        }
        return "PASSED";
    }

    private int scenarioDuration(Map<String, Object> scenario) {
        List<Number> durations = read(scenario, "$.steps[*].result.duration");
        long total = 0;
        for (Number duration : durations) {
            total = total + duration.longValue();
        }
        return (int)(total / 1_000_000);
    }

    private List<String> scenarioTags(Map<String, Object> scenario, List<String> featureTags) {
        List<String> tags = new ArrayList<>(featureTags);
        List<String> scenarioTags = readTags(scenario);
        tags.addAll(scenarioTags);
        return tags;
    }

    @Override
    public long getTimestamp() {
        return cucumberReportJsonFile.lastModified();
    }

}
