package com.xebialabs.xltest.gatling;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.UUID;
import com.google.common.base.Strings;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.domain.Importable;

public class GatlingSimulationLogImporter implements Importable {
    private OverthereFile gatlingReportDir;

    public GatlingSimulationLogImporter(OverthereFile gatlingReportDir) {
        this.gatlingReportDir = gatlingReportDir;
    }

    @Override
    public void doImport(final UUID testRunId, final EventHandler eventHandler) throws Exception {
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(gatlingReportDir.getFile("simulation.log").getInputStream()))) {
            String firstLine = bufferedReader.readLine();
            String last = null;
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                last = line;
            }
            Event startEvent = readFirstLine(testRunId, firstLine);
            storeEvents(eventHandler, startEvent);
            Event resultEvent = parseGlobalStatsReport(testRunId, gatlingReportDir);
            Event finishEvent = readLastLine(testRunId, last, (Long) startEvent.get("timestamp"));
            storeEvents(eventHandler, finishEvent, resultEvent);
        }
    }

    private Event readFirstLine(UUID testRunId, String firstLine) throws Exception {
        if (!Strings.isNullOrEmpty(firstLine)) {
            Line line = Line.parseLine(firstLine);
            return new Event(testRunId, Event.IMPORT_STARTED, Event.props("timestamp", line.getStartTime(), "simulationName", line.getSimulationName()));
        }
        return null;
    }

    private Event readLastLine(UUID testRunId, String lastLine, Long startTime) throws Exception {
        if (!Strings.isNullOrEmpty(lastLine)) {
            Line line = Line.parseLine(lastLine);
            return new Event(testRunId, Event.IMPORT_FINISHED, Event.props("timestamp", line.getEndTime(), "duration", line.getEndTime() - startTime));
        }
        return null;
    }

    private Event parseGlobalStatsReport(UUID testRunId, OverthereFile simulationReportFolder) throws Exception {
        File globalStatsJson = new File(simulationReportFolder.getPath() + File.separator + "js" + File.separator + "global_stats.json");
        GatlingGlobalStatsReport statsReport = GatlingGlobalStatsJsonParser.parse(globalStatsJson);
        return new Event(testRunId, Event.PERFORMANCE_RESULT, statsReport.toMap());
    }

    private void storeEvents(EventHandler eventHandler, Event... events) throws Exception {
        for (Event event : events) {
            eventHandler.onReceive(event);
        }
    }
}
