package com.xebialabs.xltest.jmeter;

import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;

import com.xebialabs.xltest.domain.ImportException;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVRecord;
import com.google.common.primitives.Longs;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.PerformanceResultsSummary;
import com.xebialabs.xltest.domain.SamplingPerformanceResultsSummary;
import com.xebialabs.xltest.importers.AbstractPerformanceSummaryResultsImporter;


public class JMeterCsvImporter extends AbstractPerformanceSummaryResultsImporter {

    public static final int COLUMN_TIMESTAMP = 0;
    public static final int COLUMN_ELAPSED = 1;
    public static final int COLUMN_SAMPLE_RESULT = 7;

    private final OverthereFile jmeterCsvFile;

    public JMeterCsvImporter(OverthereFile jmeterCsvFile) {
        this.jmeterCsvFile = jmeterCsvFile;
    }

    @Override
    public long getTimestamp() {
        return jmeterCsvFile.lastModified();
    }

    @Override
    public String getImportablePath() {
        return jmeterCsvFile.getPath();
    }

    @Override
    public PerformanceResultsSummary constructPerformanceResultsSummary() throws ImportException {
        boolean parsedHeader = false;

        SamplingPerformanceResultsSummary performanceResultsSummary = new SamplingPerformanceResultsSummary();
        try (Reader in = new InputStreamReader(jmeterCsvFile.getInputStream(), StandardCharsets.UTF_8)) {
            Iterable<CSVRecord> records = CSVFormat.DEFAULT.parse(in);
            for (CSVRecord record : records) {
                String sampleResult = record.get(COLUMN_SAMPLE_RESULT);
                Long timestamp = Longs.tryParse(record.get(COLUMN_TIMESTAMP));
                if (timestamp == null) {
                    if (!parsedHeader) {
                        parsedHeader = true;
                        continue;
                    } else {
                        throw new JMeterImportFailedException("Line " + record.getRecordNumber() + " does not contain a valid timestamp.", record.toString());
                    }
                }
                long elapsed = Long.parseLong(record.get(COLUMN_ELAPSED));
                performanceResultsSummary.updateWithSample(timestamp, elapsed, sampleResult);
            }
            return performanceResultsSummary;
        } catch (JMeterImportFailedException e) {
            throw e;
        } catch (Exception e) {
            throw new JMeterImportFailedException("Failed to import CSV sample data", e);
        }
    }

    @Override
    public String toString() {
        return String.format("JMeterCsvImporter(%s)", this.jmeterCsvFile);
    }
}
