package com.xebialabs.xltest.repository;

import java.util.*;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.service.EventRepository;
import com.xebialabs.xltest.utils.Pair;


@Repository
public class TestRunsRepositoryImpl implements TestRunsRepository {

    private static Logger LOG = LoggerFactory.getLogger(TestRunsRepository.class);

    private final EventRepository eventRepository;

    @Autowired
    public TestRunsRepositoryImpl(EventRepository eventRepository) {
        this.eventRepository = eventRepository;
    }

    public TestRun getTestRun(String testRunId) {
        return getTestRun(UUID.fromString(testRunId));
    }

    public TestRun getTestRun(UUID testRunId) {
        TestRun tempTestRun = new TestRun(testRunId);
        tempTestRun.setEventRepository(eventRepository);
        Event executionStartedEvent = getEventFromTestRun(tempTestRun, Event.EXECUTION_STARTED);
        Event importStartedEvent = getEventFromTestRun(tempTestRun, Event.IMPORT_STARTED);
        Event importFinishedEvent = getEventFromTestRun(tempTestRun, Event.IMPORT_FINISHED);
        Event executionFinishedEvent = getEventFromTestRun(tempTestRun, Event.EXECUTION_FINISHED);
        Event qualificationEvent = getEventFromTestRun(tempTestRun, Event.QUALIFICATION_COMPUTED);

        TestRun testRun = createTestRun(testRunId, executionStartedEvent, importStartedEvent, importFinishedEvent, executionFinishedEvent, qualificationEvent);
        testRun.setEventRepository(eventRepository);
        return testRun;
    }

    public TestRun getTestRunFromEvents(List<Event> events) {
        Event executionStartedEvent = null,
                executionFinishedEvent = null,
                importStartedEvent = null,
                importFinishedEvent = null,
                qualificationEvent = null;
        UUID testRunId = null;
        for (Event event : events) {
            String type = event.getType();
            if (testRunId == null) {
                testRunId = event.getTestRunId();
            }

            // This is assuming the events in the list all belong to the same test run id
            if (Event.EXECUTION_STARTED.equals(type) && executionStartedEvent == null) {
                executionStartedEvent = event;
            } else if (Event.EXECUTION_FINISHED.equals(type)) {
                executionFinishedEvent = event;
            } else if (Event.IMPORT_STARTED.equals(type) && importStartedEvent == null) {
                importStartedEvent = event;
            } else if (Event.IMPORT_FINISHED.equals(type)) {
                importFinishedEvent = event;
            } else if (Event.QUALIFICATION_COMPUTED.equals(type)) {
                qualificationEvent = event;
            }
        }
        return createTestRun(testRunId, executionStartedEvent, importStartedEvent, importFinishedEvent, executionFinishedEvent, qualificationEvent);
    }

    private Event getEventFromTestRun(TestRun testRun, String eventType) {
        Map<String, String> queryParameters = new HashMap<String, String>();
        queryParameters.put(Event.TYPE, eventType);
        List<Event> events = testRun.getEvents(queryParameters);
        if (events != null && events.size() >= 1) {
            return events.get(0);
        }
        return null;
    }

    /**
     * Note: Items are not auto-wired.
     *
     * @return
     */
    public Collection<TestRun> getAllTestRuns() {
        return getTestRuns(Collections.emptyMap());
    }

    private String constructQuery(long startTime, long endTime, Map queryParameters) {
        return constructSearchQuery("timestamp:>=" + startTime + " AND timestamp:<=" + endTime, queryParameters);
    }

    private String constructSearchQuery(String query, Map queryParameters) {
        for (Object key : queryParameters.keySet()) {
            String value = (String) queryParameters.get(key);
            query = query + " AND " + key + ":\"" + value + "\"";
        }
        return query;
    }

    private List<Event> getEventsBetweenByType(long startTime, long endTime, String eventType, Map queryParameters) {
        Map<String, String> startTestRunQuery = new HashMap<String, String>(queryParameters);
        startTestRunQuery.put("type", eventType);
        String query = constructQuery(startTime, endTime, startTestRunQuery);
        LOG.info("Querying ES with: {} for events in a timerange filtered by query params", query);
        return eventRepository.query(query);
    }

    public List<TestRun> getTestRuns(Map queryParameters) {
        int startTime = 0;
        long endTime = System.currentTimeMillis();
        return getTestRuns(queryParameters, startTime, endTime);
    }

    public List<TestRun> getTestRunsBetween(long startTime, long endTime) {
        return getTestRuns(Collections.emptyMap(), Math.min(startTime, endTime), Math.max(startTime, endTime));
    }

    public List<TestRun> getTestRuns(Map queryParameters, long startTime, long endTime) {
        List<Event> allStartEventsAsList = getEventsBetweenByType(startTime, endTime, Event.EXECUTION_STARTED, queryParameters);
        Map<UUID, Event> startEvents = getAllEventsAsMap(allStartEventsAsList);

        List<Event> allImportStartedEvents = getEventsBetweenByType(startTime, endTime, Event.IMPORT_STARTED, queryParameters);
        Map<UUID, Event> importStartedEvents = getAllEventsAsMap(allImportStartedEvents);

        List<Event> allImportFinishedEvents = getEventsBetweenByType(startTime, endTime, Event.IMPORT_FINISHED, queryParameters);
        Map<UUID, Event> importFinishedEvents = getAllEventsAsMap(allImportFinishedEvents);

        List<Event> allFinishTestRunEvents = getEventsBetweenByType(startTime, endTime, Event.EXECUTION_FINISHED, queryParameters);
        Map<UUID, Event> finishTestRunEvents = getAllEventsAsMap(allFinishTestRunEvents);

        List<Event> allQualificationEvents = getEventsBetweenByType(startTime, endTime, Event.QUALIFICATION_COMPUTED, queryParameters);
        Map<UUID, Event> qualificationEvents = getAllEventsAsMap(allQualificationEvents);

        Set<UUID> allUuids = getAllUuids(startEvents, importStartedEvents);

        List<TestRun> testRuns = new ArrayList<TestRun>();
        for (UUID uuid : allUuids) {
            Event startEvent = startEvents.get(uuid);
            Event importStarted = importStartedEvents.get(uuid);
            Event importFinished = importFinishedEvents.get(uuid);
            Event finishTestRunEvent = finishTestRunEvents.get(uuid);
            Event qualificationEvent = qualificationEvents.get(uuid);
            TestRun testRun = createTestRun(uuid, startEvent, importStarted, importFinished, finishTestRunEvent, qualificationEvent);
            testRun.setEventRepository(eventRepository);
            testRuns.add(testRun);
        }

        Collections.sort(testRuns, new Comparator<TestRun>() {
            @Override
            public int compare(TestRun o1, TestRun o2) {
                Date startTime1 = o1.getStartTime() != null ? o1.getStartTime() : new Date(0);
                Date startTime2 = o2.getStartTime() != null ? o2.getStartTime() : new Date(0);
                return startTime2.compareTo(startTime1);
            }
        });
        return testRuns;
    }

    private Set<UUID> getAllUuids(Map<UUID, Event> startEvents, Map<UUID, Event> importStartedEvents) {
        Set<UUID> allUuids = new HashSet<UUID>();
        allUuids.addAll(startEvents.keySet());
        allUuids.addAll(importStartedEvents.keySet());
        return allUuids;
    }

    private Map<UUID, Event> getAllEventsAsMap(List<Event> events) {
        Map<UUID, Event> map = new HashMap<UUID, Event>();
        for (Event ev : events) {
            map.put(ev.getTestRunId(), ev);
        }
        return map;
    }

    private static final Set<String> standardProps = new HashSet<String>(Arrays.asList("timestamp", "testSpecification", "runId", "type"));


    private TestRun createTestRun(UUID uuid, Event startEvent, Event importStartedEvent, Event importFinishedEvent, Event finishEvent, Event qualificationEvent) {
        TestRun testRun = new TestRun(uuid);
        if (startEvent != null) {
            testRun.setTestSpecificationName((String) startEvent.get(Event.TEST_SPECIFICATION));
            testRun.setStartTime(new Date(((Number) startEvent.get(Event.TIMESTAMP)).longValue()));
        } else {
            if (importStartedEvent != null) {
                testRun.setTestSpecificationName((String) importStartedEvent.get(Event.TEST_SPECIFICATION));
                testRun.setStartTime(new Date(((Number) importStartedEvent.get(Event.LAST_MODIFIED)).longValue()));
            } else {
                throw new NoTestRunFoundException("Can't populate TestRun from these executionStarted or importStarted events. Both could not be found");
            }
        }

        if (finishEvent != null && finishEvent.get(Event.TIMESTAMP) != null) {
            testRun.setFinishedTime(new Date(((Number) finishEvent.get(Event.TIMESTAMP)).longValue()));
        } else if (importFinishedEvent != null && importFinishedEvent.get(Event.TIMESTAMP) != null) {
            testRun.setFinishedTime(new Date(((Number) importFinishedEvent.get(Event.TIMESTAMP)).longValue()));
        }

        Map<String, Object> parameters = new HashMap<>();
        if (startEvent != null) {
            for (String key : startEvent.getProperties().keySet()) {
                if (!standardProps.contains(key)) {
                    parameters.put(key, startEvent.get(key));
                }
            }
        }

        if (finishEvent != null) {
            for (String key : finishEvent.getProperties().keySet()) {
                if (!standardProps.contains(key)) {
                    parameters.put(key, finishEvent.get(key));
                }
            }
        }

        testRun.setParameters(parameters);

        if (qualificationEvent != null) {
            testRun.setQualificationResult((Boolean) qualificationEvent.get(Event.QUALIFICATION));
            String failureReason = "";
            if (qualificationEvent.hasProperty(Event.FAILURE_REASON)) {
                failureReason = (String) qualificationEvent.get(Event.FAILURE_REASON);
            }
            testRun.setFailureReason(failureReason);
        }
        return testRun;
    }

    public List<TestRun> getPreviousRuns(TestRun testRun, int max) {
        return getSimilarRunsSortedInTime(testRun, max, getReverseComparator(), true);
    }

    public List<TestRun> getLaterRuns(TestRun testRun, int max) {
        return getSimilarRunsSortedInTime(testRun, max, getNormalComparator(), false);
    }

    public List<Event> getEventsBetween(long startTime, long endTime, Map<String, Object> eventProperties) {
        String query = constructQuery(startTime, endTime, eventProperties);
        return eventRepository.query(query);
    }

    public Date parseDateString(String dateString) {
        if (StringUtils.hasText(dateString)) {
            try {
                DateTime dateTime = new DateTime(Long.parseLong(dateString));
                return dateTime.withTimeAtStartOfDay().toDate();
            } catch (Exception e) {
                LOG.error("Unable to convert dateString (expected to be timestamp) [" + dateString + "] into Date.");
            }
        }
        return null;
    }

    public Date makeStartDateIfProvided(String dateString) {
        Date result = parseDateString(dateString);
        if (result == null) {
            result = new DateTime().minusWeeks(2).withTimeAtStartOfDay().toDate();
        }
        return result;
    }

    public Date makeEndDateIfProvided(String dateString) {
        Date result = parseDateString(dateString);
        if (result == null) {
            result = new Date();
        }
        return result;
    }

    private List<TestRun> getSimilarRunsSortedInTime(TestRun testRun, int max, Comparator<TestRun> comparator, boolean before) {
        Collection<TestRun> allTestRuns = getAllTestRuns();
        String testSpecificationId = testRun.getTestSpecificationName();
        List<TestRun> allTestRunsWithTheSameReference = new ArrayList<TestRun>();
        for (TestRun tr : allTestRuns) {
            if (testSpecificationId.equals(tr.getTestSpecificationName())) {
                allTestRunsWithTheSameReference.add(tr);
            }
        }

        Collections.sort(allTestRunsWithTheSameReference, comparator);
        // get the last <max> that were before this run itself
        List<TestRun> similarRunsSortedInTime = new ArrayList<TestRun>();
        Date startTime = testRun.getStartTime();
        int cnt = 0;
        for (TestRun tr : allTestRunsWithTheSameReference) {
            if (beforeOrAfter(startTime, tr, before)) {
                similarRunsSortedInTime.add(tr);
                cnt++;
            }
            if (cnt >= max) {
                break;
            }
        }
        LOG.debug("I got " + similarRunsSortedInTime.size() + " similar runs");
        return similarRunsSortedInTime;
    }

    private boolean beforeOrAfter(Date startTime, TestRun testRun, boolean before) {
        if (before) {
            return testRun.getStartTime().before(startTime);
        } else {
            return testRun.getStartTime().after(startTime);
        }
    }

    private Comparator<TestRun> getNormalComparator() {
        Comparator<TestRun> comparator = new Comparator<TestRun>() {
            @Override
            public int compare(TestRun r1, TestRun r2) {
                return r1.getStartTime().compareTo(r2.getStartTime());
            }
        };
        return comparator;
    }

    private Comparator<TestRun> getReverseComparator() {
        Comparator<TestRun> comparator = new Comparator<TestRun>() {
            @Override
            public int compare(TestRun r1, TestRun r2) {
                return r2.getStartTime().compareTo(r1.getStartTime());
            }
        };
        return comparator;
    }

    public List<Event> getCorrespondingEventsFromOlderRuns(TestRun testRun, Event event, int max, String... matchingProperties) {
        List<Event> events = new ArrayList<Event>();
        List<TestRun> previousRuns = getPreviousRuns(testRun, max);
        LOG.debug("While getting events from max " + max + " previous Runs, I got " + previousRuns.size() + " such runs");
        Map queryParameters = makeQueryParameters(event, matchingProperties);

        for (TestRun previousRun : previousRuns) {
            List<Event> eventsOfPreviousRun = previousRun.getEvents(queryParameters);
            LOG.debug("Found " + eventsOfPreviousRun.size() + " events with matching properties in a previous run");
            events.addAll(eventsOfPreviousRun);
        }
        LOG.debug("Found " + events.size() + " CorrespondingEventsFromOlderRuns.");
        return events;
    }

    private Map makeQueryParameters(Event event, String... matchingProperties) {
        Map queryParameters = new HashMap();
        if (matchingProperties != null && matchingProperties.length != 0) {
            for (int i = 0; i < matchingProperties.length; i++) {
                String propertyKey = matchingProperties[i];
                queryParameters.put(propertyKey, event.get(propertyKey));
            }
        }
        return queryParameters;
    }

    public List<TestRun> getPreviousRunsIncludingThisOne(TestRun testRun, int max) {
        List<TestRun> runs = getPreviousRuns(testRun, max);
        runs.add(testRun);
        return runs;
    }


    public List<Event> getCorrespondingEventsFromOtherRuns(Event event, List<TestRun> otherRuns, String... matchingProperties) {
        List<Event> events = new ArrayList<Event>();
        Map queryParameters = makeQueryParameters(event, matchingProperties);
        for (TestRun previousRun : otherRuns) {
            List<Event> eventsOfPreviousRun = previousRun.getEvents(queryParameters);
            LOG.debug("Found " + eventsOfPreviousRun.size() + " events with matching properties in a previous run");
            events.addAll(eventsOfPreviousRun);
        }
        LOG.debug("Found " + events.size() + " CorrespondingEventsFromOlderRuns");
        return events;
    }

    //getEventsFromOtherRunsAsMapGroupedBy('suiteName', olderRuns, 'type:jobStatus', 'status:finished')
    public Map<String, List<Event>> getEventsFromRunsAsMapGroupedBy(String mapProperty, List<TestRun> runs, String... matchingKVs) {
        Map<String, Object> properties = makeEventProperties(matchingKVs);
        Set<String> keySet = properties.keySet();
        String[] matchingProperties = keySet.toArray(new String[keySet.size()]); // 'type', 'status'
        Event event = new Event(properties);
        List<Event> allEvents = getCorrespondingEventsFromOtherRuns(event, runs, matchingProperties);
        //now group event by 'mapProperty'
        Map<String, List<Event>> map = new HashMap<String, List<Event>>();
        for (Event e : allEvents) {
            String groupByValue = e.get(mapProperty);
            if (map.get(groupByValue) == null) {
                map.put(groupByValue, new ArrayList<Event>());
            }
            map.get(groupByValue).add(e);
        }
        return orderEventsByTimeStampNewestOnTop(map);
    }

    public Map<String, List<Event>> getEventsFromRunsAsMapGroupedBy(String mapProperty, int level, List<TestRun> runs, String... matchingKVs) {
        Map<String, Object> properties = makeEventProperties(matchingKVs);
        Set<String> keySet = properties.keySet();
        String[] matchingProperties = keySet.toArray(new String[keySet.size()]); // 'type', 'status'
        Event event = new Event(properties);
        List<Event> allEvents = getCorrespondingEventsFromOtherRuns(event, runs, matchingProperties);
        //now group event by 'mapProperty'
        Map<String, List<Event>> map = new HashMap<String, List<Event>>();
        for (Event e : allEvents) {
            String groupByValue = cutAtLevel((String) e.get(mapProperty), level);
            if (map.get(groupByValue) == null) {
                map.put(groupByValue, new ArrayList<Event>());
            }
            map.get(groupByValue).add(e);
        }
        return orderEventsByTimeStampNewestOnTop(map);
    }

    private Map<String, List<Event>> orderEventsByTimeStampNewestOnTop(Map<String, List<Event>> map) {
        // order events by ts, the newest up front
        for (String key : map.keySet()) {
            Collections.sort(map.get(key), new Comparator<Event>() {
                @Override
                public int compare(Event r1, Event r2) {
                    Comparable t1 = null, t2 = null;
                    try {
                        t1 = r1.get(Event.TIMESTAMP);
                        t2 = r2.get(Event.TIMESTAMP);
                        return t1.compareTo(t2);
                    } catch (Exception e) {
                        // Catch ClassCastException and possible NullPointerExceptions...
                        LOG.error("t1: {} can't be compared to {}. t1 class: {}", t1, t2);
                        return 0;
                    }
                }
            });
        }
        return map;
    }

    private String cutAtLevel(String pageName, int level) {
        String[] parts = pageName.split(";"); // old: "\\."
        if (parts == null || level > parts.length) {
            return pageName;
        }
        int index = 0;
        for (int i = 0; i < level; i++) {
            index = index + parts[i].length() + 1;
        }
        String result = pageName.substring(0, index);
        if (result.endsWith(";")) {
            result = result.substring(0, result.length() - 1);
        }
        return result;
    }

    private Map<String, Object> makeEventProperties(String[] matchingKVs) {
        // 'type:jobStatus', 'status:finished'
        Map<String, Object> eventProperties = new HashMap<String, Object>();
        for (int i = 0; i < matchingKVs.length; i++) {
            Pair p = parseKV(matchingKVs[i]);
            eventProperties.put(p.getKey(), p.getValue());
        }
        return eventProperties;
    }

    private Pair parseKV(String kv) {
        // key:value
        String key = kv.substring(0, kv.indexOf(':'));
        String value = kv.substring(key.length() + 1);
        return new Pair(key, value);
    }
}
