package com.xebialabs.xltest.resources;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.*;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.*;

import com.xebialabs.deployit.plugin.overthere.Host;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.collect.Lists;
import com.jayway.jsonpath.spi.JsonProviderFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestSpecification;
import com.xebialabs.xltest.service.EventRepository;

import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;


@Controller
@Path("/migrate")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class MigrationController {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationController.class);

    private static final String[] propsToCopy = {"branch", "browser", "environment", "repositoryUrl", "suiteFilter", "suiteName", "webdriver"};


    private final EventRepository eventRepository;
    private final RepositoryService repository;

    private static boolean alreadyMigrated = false;

    @Autowired
    public MigrationController(EventRepository eventRepository, RepositoryService repository) {
        this.eventRepository = eventRepository;
        this.repository = repository;
    }

    @GET
    @Path("/")
    public Response migrate(@Context UriInfo uriInfo) {
        String feedback = "";
        try {
            MultivaluedMap<String, String> queryParameters = uriInfo.getQueryParameters();
            String host = null;
            if (queryParameters.get("host") != null) {
                host = queryParameters.get("host").get(0);
            }
            String port = null;
            if (queryParameters.get("port") != null) {
                port = queryParameters.get("port").get(0);
            }
            if (host != null && port != null && !alreadyMigrated) {
                URL oldBaseUrl = new URL("http://" + host + ":" + port); // eg. http://localhost:9400
                if (checkMigratable(oldBaseUrl)) {
                    LOG.info("migration started");
                    alreadyMigrated = true;
                    migrateJCRReports();
                    feedback = migrateES(oldBaseUrl);
                } else {
                    feedback = "Sorry, can't migrate as I can't reach your old ElasticSearch on host " + host + " and port " + port;
                }
            } else {
                if (alreadyMigrated) {
                    feedback = "Sorry, can't migrate. It seems you already did. Restart server when in doubt";
                } else {
                    feedback = "Sorry, can't migrate. I expected two query parameters: host=<hostname> and port=<portnumber>";
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(400).build();
        }
        return Response.ok(feedback).build();
    }

    private void migrateJCRReports() {
        Report compositePerTeam = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        compositePerTeam.setId("Configuration/Reports/CompositePerTeam");
        compositePerTeam.setProperty("title", "Test Results per team");
        compositePerTeam.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(compositePerTeam);
        Report bar = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        bar.setId("Configuration/Reports/barchart");
        bar.setProperty("scriptLocation", "reports/GraphReport.py");
        bar.setProperty("drilldownsequence", "team, application, applicationPackage, usecase, testcase");
        bar.setProperty("drilldown", compositePerTeam);
        bar.setProperty("useTeamInDrillDown", true);
        repository.createOrUpdate(bar);
        Report pie = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.PieReport")).newInstance("");
        pie.setId("Configuration/Reports/pie");
        pie.setProperty("scriptLocation", "reports/PieReport.py");
        repository.createOrUpdate(pie);
        Report details = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        details.setId("Configuration/Reports/drilldown");
        details.setProperty("scriptLocation", "reports/drilldowndetails.ftl");
        details.setProperty("reportType", "html");
        details.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(details);
        compositePerTeam.setProperty("topLeft", bar);
        compositePerTeam.setProperty("topRight", pie);
        compositePerTeam.setProperty("bottom", details);
        repository.createOrUpdate(compositePerTeam);

        Report compositePerApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        compositePerApp.setId("Configuration/Reports/CompositePerApp");
        compositePerApp.setProperty("title", "Test Results");
        compositePerApp.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(compositePerApp);
        Report barchartPerApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        barchartPerApp.setId("Configuration/Reports/barchartPerApp");
        barchartPerApp.setProperty("scriptLocation", "reports/GraphReport.py");
        barchartPerApp.setProperty("drilldownsequence", "application, applicationPackage, usecase, testcase");
        barchartPerApp.setProperty("drilldown", compositePerApp);
        repository.createOrUpdate(barchartPerApp);
        compositePerApp.setProperty("topLeft", barchartPerApp);
        compositePerApp.setProperty("topRight", pie);
        compositePerApp.setProperty("bottom", details);
        repository.createOrUpdate(compositePerApp);

        Report monitor = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        monitor.setId("Configuration/Reports/monitor");
        monitor.setProperty("title", "Test Automation Results");
        monitor.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(monitor);

        Report monitorApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        monitorApp.setId("Configuration/Reports/monitorApp");
        monitorApp.setProperty("title", "Test Automation Results Per App");
        monitorApp.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(monitorApp);

        Report monitorChart = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorChart.setId("Configuration/Reports/Monitor chart");
        monitorChart.setProperty("scriptLocation", "reports/DashboardReport.py");
        monitorChart.setProperty("drilldownsequence", "team, application, applicationPackage, usecase");
        monitorChart.setProperty("drilldown", monitor);
        monitorChart.setProperty("useTeamInDrillDown", true);
        repository.createOrUpdate(monitorChart);

        Report monitorAppChart = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorAppChart.setId("Configuration/Reports/Monitor APP chart");
        monitorAppChart.setProperty("scriptLocation", "reports/DashboardReport.py");
        monitorAppChart.setProperty("drilldownsequence", "team, application, usecase, testcase");
        monitorAppChart.setProperty("drilldown", monitorApp);
        repository.createOrUpdate(monitorAppChart);

        Report monitorPie = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorPie.setId("Configuration/Reports/Monitor pie chart");
        monitorPie.setProperty("scriptLocation", "reports/DashboardPieReport.py");
        monitorPie.setProperty("drilldownsequence", "team, application, applicationPackage, usecase, testcase");
        monitorPie.setProperty("drilldown", monitor);
        repository.createOrUpdate(monitorPie);

        Report monitorNavigation = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        monitorNavigation.setId("Configuration/Reports/Monitor navigation chart");
        monitorNavigation.setProperty("scriptLocation", "reports/monitorBottom.ftl");
        monitorNavigation.setProperty("reportType", "html");
        monitorNavigation.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(monitorNavigation);

        Report monitorAppNavigation = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        monitorAppNavigation.setId("Configuration/Reports/Monitor APP navigation chart");
        monitorAppNavigation.setProperty("scriptLocation", "reports/monitorBottomApp.ftl");
        monitorAppNavigation.setProperty("reportType", "html");
        monitorAppNavigation.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(monitorAppNavigation);

        monitor.setProperty("topLeft", monitorChart);
        monitor.setProperty("topRight", monitorPie);
        monitor.setProperty("bottom", monitorNavigation);
        repository.createOrUpdate(monitor);

        monitorApp.setProperty("topLeft", monitorAppChart);
        monitorApp.setProperty("topRight", monitorPie);
        monitorApp.setProperty("bottom", monitorAppNavigation);
        repository.createOrUpdate(monitorApp);

        Report jobsPerSlave = DescriptorRegistry.getDescriptor(Type.valueOf("freemarker.Report")).newInstance("");
        jobsPerSlave.setId("Configuration/Reports/jobsPerSlave");
        jobsPerSlave.setProperty("scriptLocation", "reports/jobsPerSlave.ftl");
        jobsPerSlave.setProperty("reportType", "html");
        jobsPerSlave.setProperty("iconName", "none");
        jobsPerSlave.setProperty("userFriendlyDescription", "Jobs per slave report");
        jobsPerSlave.setProperty("applicableTools", Lists.newArrayList("bol.SlicedFitNesse", "bol.SlicedFitNesseSubSet"));
        repository.createOrUpdate(jobsPerSlave);

        Report freemarkerDuration = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        freemarkerDuration.setId("Configuration/Reports/BolDuration");
        freemarkerDuration.setProperty("scriptLocation", "reports/durationdetails.ftl");
        freemarkerDuration.setProperty("reportType", "html");
        freemarkerDuration.setProperty("iconName", "none");
        freemarkerDuration.setProperty("userFriendlyDescription", "Tabular duration report");
        freemarkerDuration.setProperty("maxRunsInHistory", 20);
        repository.createOrUpdate(freemarkerDuration);
    }

    private boolean checkMigratable(URL oldBaseUrl) {
        HttpURLConnection connToOldES = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_mapping");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();

            LOG.info("Connected to old ESs. Resp. code on old: " + responseCodeOnOldMappingUrl);

            if (responseCodeOnOldMappingUrl == 200) {
                // we assume we're all set and off we go
            } else {
                LOG.error("Could not connect to one or both ES instances, see above.....");
                return false;
            }

        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
        }
        LOG.info("Found old ES so we have a migratable situation");
        return true;
    }

    private String migrateES(URL oldBaseUrl) {
        long now = System.currentTimeMillis();
        Map<String, List<Event>> jobStatusEventsPerRunId = migrateRuns(oldBaseUrl);
        int nResults = migrateResults(oldBaseUrl, jobStatusEventsPerRunId);
        long durationInMillies = System.currentTimeMillis() - now;
        long durationInSec = durationInMillies / 1000;
        long durationInMin = durationInSec / 60;
        long durationInHrs = durationInMin / 60;
        durationInMin = durationInMin - (durationInHrs * 60);
        durationInSec = (durationInSec - (durationInHrs * 60 * 60)) - (durationInMin * 60);
        String feedback = "Migration done in " + durationInHrs + " hrs " + durationInMin + " min " + durationInSec + " sec, and there were " + nResults + " test result events found in " + jobStatusEventsPerRunId.keySet().size() + " test runs";
        LOG.info(feedback);
        return feedback;
    }

    private Map<String, List<Event>> migrateRuns(URL oldBaseUrl) {
        LOG.info("Searching jobStatus events. This may take a while as there are hundreds of thousands of them....");
        Map<String, List<Event>> jobStatusEventsPerRunId = getJobStatusEventsPerRunId(oldBaseUrl);
        LOG.info("I found " + jobStatusEventsPerRunId.keySet().size() + " run_ids in jobStatus events");

        List<Event> startTestRunEvents = getStartOrFinishTestRunEvents(oldBaseUrl, "startTestRun", jobStatusEventsPerRunId);
        List<Event> finishTestRunEvents = getStartOrFinishTestRunEvents(oldBaseUrl, "finishTestRun", jobStatusEventsPerRunId);
        Set<String> runIdsStartedButNotFinished = getRunIdsStartedButNotFinished(startTestRunEvents, finishTestRunEvents);
        LOG.info("I found " + startTestRunEvents.size() + " start events and " + finishTestRunEvents.size() + " finish events");


        List<Event> additionalFinishTestRunEvents = compensateMissingFinishes(runIdsStartedButNotFinished, jobStatusEventsPerRunId);
        LOG.info("I found " + additionalFinishTestRunEvents.size() + " additional finish events after compensating");
        LOG.info("Now I have started: " + startTestRunEvents.size() + " start events " +
                (finishTestRunEvents.size() + additionalFinishTestRunEvents.size()) + " finish events in total");
        int totalInserted = 0;
        int inserted = 0;
        inserted = insertEvents(startTestRunEvents, " start run events");
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " start events. Total of inserted events is now: " + totalInserted);
        inserted = insertEvents(finishTestRunEvents, " finish run events");
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " finish events. Total of inserted events is now: " + totalInserted);
        inserted = insertCorrespondingImportEvents(finishTestRunEvents);
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " corresponding import events. Total of inserted events is now: " + totalInserted);
        inserted = insertEvents(additionalFinishTestRunEvents, " additional finish run events");
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " additional finish events. Total of inserted events is now: " + totalInserted);
        inserted = insertCorrespondingImportEvents(additionalFinishTestRunEvents);
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " additional corresponding import events. Total of inserted events is now: " + totalInserted);
        inserted = insertEvents(jobStatusEventsPerRunId);
        totalInserted = totalInserted + inserted;
        LOG.info("Inserted " + inserted + " jobStatus events. Total of inserted events is now: " + totalInserted);
        return jobStatusEventsPerRunId;
    }

    private int insertCorrespondingImportEvents(List<Event> finishRunEvents) {
        // some reports take import events as point of departure. These are lacking in the old DB
        // so we create an import event based on all finishedEvents
        List<Event> importEvents = new ArrayList<Event>();
        for (Event finishRunEvent : finishRunEvents) {
            Map<String, Object> otherProperties = new HashMap<String, Object>();
            otherProperties.put("runId", finishRunEvent.get("runId"));
            otherProperties.put("timestamp", finishRunEvent.get("timestamp"));
            otherProperties.put("lastModified", finishRunEvent.get("timestamp"));
            otherProperties.put("testSpecification", finishRunEvent.get("testSpecification"));
            Event importEvent = new Event("importStarted", otherProperties);
            importEvents.add(importEvent);
        }
        return insertEvents(importEvents, " import events");
    }

    private int insertEvents(Map<String, List<Event>> jobStatusEventsPerRunId) {
        int cnt = 0;
        for (String runid : jobStatusEventsPerRunId.keySet()) {
            cnt = cnt + insertEvents(jobStatusEventsPerRunId.get(runid), " jobStatus events for run " + runid);
        }
        return cnt;
    }

    private int insertEvents(List<Event> events, String progressInfo) {
        boolean printProgress = events.size() > 2000;
        int cnt = 0;
        for (Event event : events) {
            insertEvent(event);
            if (printProgress) {
                cnt++;
                if (cnt % 1000 == 999) {
                    LOG.info("Inserted " + (cnt + 1) + " events out of " + events.size() + progressInfo);
                }
            }
        }
        return cnt;
    }

    private void insertEvent(Event event) {
        try {
            eventRepository.insert(event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private List<Event> compensateMissingFinishes(Set<String> runIdsStartedButNotFinished, Map<String, List<Event>> jobStatusEventsPerRunId) {
        List<Event> missingFinishEvents = new ArrayList<Event>();
        for (String runId : runIdsStartedButNotFinished) {
            if (jobStatusEventsPerRunId.get(runId) != null) {
                Event lastFinishedJobStatus = getLastFinishTimeEvent(jobStatusEventsPerRunId.get(runId));
                Map<String, Object> otherProperties = new HashMap<String, Object>();
                otherProperties.put("testSpecification", convertToSimpleName((String) lastFinishedJobStatus.get("testSpecification")));
                otherProperties.put("runId", runId);
                otherProperties.put("timestamp", lastFinishedJobStatus.get("timestamp"));
                // get parameters from any job status event and put them in finished event
                addParameters(otherProperties, jobStatusEventsPerRunId.get(runId).get(0));
                Event compensatedFinishEvent = new Event("executionFinished", otherProperties);
                missingFinishEvents.add(compensatedFinishEvent);
            }
        }
        return missingFinishEvents;
    }


    private String convertToSimpleName(String testSpecification) {
        if (testSpecification != null && testSpecification.startsWith("Configuration/TestSetDefinitions/")) {
            return testSpecification.substring(testSpecification.lastIndexOf("/") + 1);
        }
        return testSpecification;
    }

    private Event getLastFinishTimeEvent(List<Event> list) {
        Event lastFinishTimeEvent = null;
        for (Event jobStatus : list) {
            if (lastFinishTimeEvent == null) {
                lastFinishTimeEvent = jobStatus;
            } else {
                String lastFinishTimeEventFinishTime = (String) lastFinishTimeEvent.get("finishedTime");
                String jobStatusFinishTime = (String) jobStatus.get("finishedTime");
                if (lastFinishTimeEventFinishTime != null && jobStatusFinishTime != null && Long.parseLong(lastFinishTimeEventFinishTime) < Long.parseLong(jobStatusFinishTime)) {
                    lastFinishTimeEvent = jobStatus;
                }
            }
        }
        return lastFinishTimeEvent;
    }

    private Set<String> getRunIdsStartedButNotFinished(List<Event> startTestRunEvents, List<Event> finishTestRunEvents) {
        Set<String> startedRunIds = getRunIdsFromEvents(startTestRunEvents);
        Set<String> finishedRunIds = getRunIdsFromEvents(finishTestRunEvents);
        startedRunIds.removeAll(finishedRunIds);
        return startedRunIds;
    }

    private Set<String> getRunIdsFromEvents(List<Event> startTestRunEvents) {
        Set<String> runIds = new HashSet<String>();
        for (Event ev : startTestRunEvents) {
            String runId = ev.get("runId");
            runIds.add(runId);
        }
        return runIds;
    }

    private List<Event> getStartOrFinishTestRunEvents(URL oldBaseUrl, String startOrFinish, Map<String, List<Event>> jobStatusEventsPerRunId) {
        List<Event> startTestRunEvents = new ArrayList<Event>();
        HttpURLConnection connToOldES = null;
        HttpURLConnection connToNewES = null;
        InputStream inputStream = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:" + startOrFinish + "&size=1000000");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
            if (responseCodeOnOldMappingUrl == 200) {
                inputStream = connToOldES.getInputStream();
                JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                JSONObject hitsAsObject = (JSONObject) result.get("hits");
                JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                for (int i = 0; i < realHits.size(); i++) {
                    JSONObject startOrFinishTestRun = (JSONObject) realHits.get(i);
                    JSONObject source = (JSONObject) startOrFinishTestRun.get("_source");
                    Map<String, Object> otherProperties = new HashMap<String, Object>();
                    for (String key : source.keySet()) {
                        if (key.equals("testSetId")) {
                            String convertToSimpleName = convertToSimpleName((String) source.get(key));
                            otherProperties.put("testSpecification", convertToSimpleName);
                        }
                        if (key.equals("run_id")) {
                            otherProperties.put("runId", source.get(key));
                        }
                        if (key.equals("_ts")) {
                            otherProperties.put("timestamp", source.get(key));
                        }
                    }
                    // get parameters from any job status event and put them in finished event
                    if (jobStatusEventsPerRunId.get(source.get("run_id")) != null) {
                        addParameters(otherProperties, jobStatusEventsPerRunId.get(source.get("run_id")).get(0));
                    }
                    String modernType = "executionStarted";
                    if ("finishTestRun".equals(startOrFinish)) {
                        modernType = "executionFinished";
                    }
                    Event modernEvent = new Event(modernType, otherProperties);
                    startTestRunEvents.add(modernEvent);
                }
            } else {
                LOG.error("Could not retrieve instances of the " + startOrFinish + " type");
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
            if (connToNewES != null) connToNewES.disconnect();
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
        }
        return startTestRunEvents;
    }

    private void addParameters(Map<String, Object> properties, Event event) {
        // event contains parameters. add these to the properties
        for (int i = 0; i < propsToCopy.length; i++) {
            if (event.hasProperty(propsToCopy[i])) {
                properties.put(propsToCopy[i], event.get(propsToCopy[i]));
            }
        }
    }

    private Map<String, List<Event>> getJobStatusEventsPerRunId(URL oldBaseUrl) {
        Map<String, List<Event>> jobStatusEventsPerRunId = new HashMap<String, List<Event>>();
        Set<String> alreadyCreatedSpecifications = new HashSet<>();
        HttpURLConnection connToOldES = null;
        InputStream inputStream = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:jobStatus&size=1000000");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
            if (responseCodeOnOldMappingUrl == 200) {
                inputStream = connToOldES.getInputStream();
                JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                JSONObject hitsAsObject = (JSONObject) result.get("hits");
                JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                LOG.info("Have to process " + realHits.size() + " jobStatus Events");
                for (int i = 0; i < realHits.size(); i++) {
                    JSONObject startOrFinishTestRun = (JSONObject) realHits.get(i);
                    JSONObject source = (JSONObject) startOrFinishTestRun.get("_source");
                    Map<String, Object> otherProperties = new HashMap<String, Object>();
                    if (i % 1000 == 999) {
                        LOG.info("Processed " + (i + 1) + " jobStatus Events");
                    }
                    for (String key : source.keySet()) {
                        if (key.equals("testSetDefinition")) {
                            otherProperties.put("testSpecification", convertToSimpleName((String) (source.get(key))));
                        }
                        if (key.equals("runId")) {
                            otherProperties.put("runId", source.get(key));
                        }
                        if (key.equals("run_id")) {
                            otherProperties.put("runId", source.get(key));
                        }
                        if (key.equals("_ts")) {
                            otherProperties.put("timestamp", source.get(key));
                        }
                        if (key.equals("startTime")) {
                            otherProperties.put("startTime", source.get(key));
                        }
                        if (key.equals("finishedTime")) {
                            otherProperties.put("finishedTime", source.get(key));
                        }
                        if (key.equals("status")) {
                            otherProperties.put("status", source.get(key));
                            if (source.get("status").equals("finished")) {
                                Number started = (Number) source.get("started");
                                long duration = 0;
                                if (started != null && source.get("_ts") != null) {
                                    duration = ((Number) source.get("_ts")).longValue() - started.longValue();
                                }
                                otherProperties.put("started", started);
                                otherProperties.put("duration", duration);
                                otherProperties.put(Event.IMPORT_FINISHED_RESULT, source.get("reason"));
                                otherProperties.put(Event.IMPORT_SLAVE, source.get("slave"));

                                otherProperties.put("suiteName", source.get("suiteName"));
                                otherProperties.put("reason", source.containsKey("reason") ? source.get("reason") : "");
                                otherProperties.put("jenkinsUri", source.containsKey("jenkinsUri") ? source.get("jenkinsUri") : "");
                                otherProperties.put("jobName", source.containsKey("jobName") ? source.get("jobName") : "");
                                otherProperties.put("slave", source.containsKey("slave") ? source.get("slave") : "");
                                otherProperties.put("timeout", source.containsKey("timeout") ? source.get("timeout") : "");
                                otherProperties.put("ntests", source.get("ntests"));

                                if (source.containsKey("buildNumber")) {
                                    otherProperties.put("buildNumber", source.get("buildNumber"));
                                }
                                if (source.containsKey("browser")) {
                                    otherProperties.put("browser", source.get("browser"));
                                }
                                if (source.containsKey("team")) {
                                    otherProperties.put("team", source.get("team"));
                                }
                                if (source.containsKey("environment")) {
                                    otherProperties.put("environment", source.get("environment"));
                                }
                                if (source.containsKey("testset")) {
                                    otherProperties.put("name", semicolonSeparate((String) source.get("testset")));
                                }
                                if (source.containsKey("uri")) {
                                    otherProperties.put("uri", source.get("uri"));
                                }
                            }
                        }
                    }

                    Event jobStatusEvent = new Event("jobStatus", otherProperties);
                    if (jobStatusEventsPerRunId.get(jobStatusEvent.get("runId")) == null) {
                        jobStatusEventsPerRunId.put(jobStatusEvent.get("runId").toString(), new ArrayList<Event>());
                    }

                    if (jobStatusEvent.hasProperty("testSpecification")) {
                        String testSpecification = jobStatusEvent.get("testSpecification");
                        if (!alreadyCreatedSpecifications.contains(testSpecification)) {
                            Host host = null;
                            if (jobStatusEvent.hasProperty("jenkinsUri") && jobStatusEvent.hasProperty("jobName")) {
                                host = createJenkinsHostInJcr((String) jobStatusEvent.get("jenkinsUri"), testSpecification, (String) jobStatusEvent.get("jobName"));
                            }
                            createTestSpecificationInJcr(testSpecification, host);
                            alreadyCreatedSpecifications.add(testSpecification);
                        }
                    }
                    jobStatusEventsPerRunId.get(jobStatusEvent.get("runId").toString()).add(jobStatusEvent);
                }
            } else {
                LOG.error("Could not retrieve instances of the jobStatus type");
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
        }
        return jobStatusEventsPerRunId;
    }

    private Host createJenkinsHostInJcr(String jenkinsUri, String testSpecification, String jobName) {
        Host host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.JenkinsHost")).newInstance("");
        host.setId("Infrastructure/" + testSpecification + " Host");
        // set minimal amount of properties: just those needed for reporting
        host.setProperty("address", jenkinsUri);
        host.setProperty("jobName", jobName);
        repository.createOrUpdate(host);
        LOG.info("New jenkins host created in JCR " + host.getId());
        return host;
    }

    private void createTestSpecificationInJcr(String testSpecificationName, Host host) {
        TestSpecification ts = DescriptorRegistry.getDescriptor(Type.valueOf("bol.SlicedFitNesse")).newInstance("");
        ts.setId("Configuration/TestSpecifications/" + testSpecificationName);
        ts.setTestToolName("FitNesse");
        if (host != null) {
            ts.setHost(host);
        }
        repository.createOrUpdate(ts);
        LOG.info("New test specification created in JCR " + ts.getId());
    }

    private int migrateResults(URL oldBaseUrl, Map<String, List<Event>> jobStatusEventsPerRunId) {
        HttpURLConnection connToOldES = null;
        InputStream inputStream = null;
        int cnt = 0;
        try {
            for (String runId : jobStatusEventsPerRunId.keySet()) {
                URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:result%20AND%20run_id:\"" + runId + "\"&size=1000000");
                connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
                int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
                if (responseCodeOnOldMappingUrl == 200) {
                    inputStream = connToOldES.getInputStream();
                    JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                    JSONObject hitsAsObject = (JSONObject) result.get("hits");
                    JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                    LOG.info("Have to process " + realHits.size() + " Result Events");
                    // the old result events do not have "testSpecification", so we stick it in below.
                    // we derive it from any jobStatus event
                    String testSpecification = jobStatusEventsPerRunId.get(runId).get(0).get("testSpecification");
                    for (int i = 0; i < realHits.size(); i++) {
                        JSONObject resultEvent = (JSONObject) realHits.get(i);
                        JSONObject source = (JSONObject) resultEvent.get("_source");
                        Map<String, Object> otherProperties = new HashMap<String, Object>();
                        if (i % 1000 == 999) {
                            LOG.info("Processed " + (i + 1) + " result Events");
                        }
                        // 13:36:56: functionalResult ->
                        // {"duration":38,"name":"DemoSuite;Backend;UseCase001;TestCase002","result":"PASSED","right":3,
                        // "runId":"70a03821-85e9-11e4-a225-c8e0eb15ab7f","testSpecification":"demoFitNesse",
                        //"timestamp":1418647016000,"type":"functionalResult","wrong":0}
                        for (String key : source.keySet()) {
                            if (key.equals("result")) {
                                otherProperties.put("result", source.get(key));
                            }
                            if (key.equals("buildNumber")) {
                                otherProperties.put("buildNumber", source.get(key));
                            }
                            if (key.equals("issue")) {
                                otherProperties.put("issue", source.get(key));
                            }
                            if (key.equals("issue")) {
                                otherProperties.put("issue", source.get(key));
                            }
                            if (key.equals("testcasedescription")) {
                                otherProperties.put("testcasedescription", source.get(key));
                            }
                            if (key.equals("buildUrl")) {
                                otherProperties.put("buildUrl", source.get(key));
                            }
                            if (key.equals("priority")) {
                                otherProperties.put("priority", source.get(key));
                            }
                            if (key.equals("browser")) {
                                otherProperties.put("browser", source.get(key));
                            }
                            if (key.equals("sprint")) {
                                otherProperties.put("sprint", source.get(key));
                            }
                            if (key.equals("_ts")) {
                                otherProperties.put("timestamp", source.get(key));
                            }
                            if (key.equals("tags")) {
                                otherProperties.put("tags", source.get(key));
                            }
                            if (key.equals("pageName")) {
                                otherProperties.put("pageName", source.get(key));
                            }
                            if (key.equals("application")) {
                                otherProperties.put("application", source.get(key));
                            }
                            if (key.equals("firstError")) {
                                otherProperties.put("firstError", source.get(key));
                            }
                            if (key.equals("usecase")) {
                                otherProperties.put("usecase", source.get(key));
                            }
                            if (key.equals("properties")) {
                                otherProperties.put("properties", source.get(key));
                            }
                            if (key.equals("team")) {
                                otherProperties.put("team", source.get(key));
                            }
                            if (key.equals("testcase")) {
                                otherProperties.put("testcase", source.get(key));
                            }
                            if (key.equals("applicationPackage")) {
                                otherProperties.put("applicationPackage", source.get(key));
                            }
                            if (key.equals("environment")) {
                                otherProperties.put("environment", source.get(key));
                            }
                            if (key.equals("run_id")) {
                                otherProperties.put("runId", source.get(key));
                            }
                            if (key.equals("testset")) {
                                otherProperties.put("name", semicolonSeparate((String) source.get(key)));
                            }
                            if (key.equals("jobName")) {
                                otherProperties.put("jobName", source.get(key));
                            }
                        }
                        // property 'duration' needs to be included otherwise our reports break.
                        // Note the exact info is not present in the old database
                        otherProperties.put("duration", 0);
                        // next, there is no "testSpecification" in the old results, needed by reports, so we slip it in (derived from the run)
                        otherProperties.put("testSpecification", testSpecification);
                        // properties 'right' and 'wrong' have been removed in the listener, so can't set those two!
                        cnt++;

                        Event functionalResult = new Event("functionalResult", otherProperties);
                        eventRepository.insert(functionalResult);
                    }
                } else {
                    LOG.error("Could not retrieve instances of the jobStatus type");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
        }
        return cnt;
    }

    private String semicolonSeparate(String fitNesseTestCaseName) {
        return fitNesseTestCaseName.replaceAll("\\.", ";");
    }


}

