package com.xebialabs.xltest.repository;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OperatingSystemFamily;

import static java.lang.String.format;

@Repository
public class HostRepository {

    public static final String HOST_TYPE_OVERTHERE_JENKINS = "overthere.JenkinsHost";

    private final RepositoryService repository;

    @Autowired
    public HostRepository(RepositoryService repositoryService) {
        this.repository = repositoryService;
    }

    public Host findJenkinsHost(String addressToLookFor) {
        return findHost(addressToLookFor, 0, HOST_TYPE_OVERTHERE_JENKINS);
    }

    public Host findHost(String addressToLookFor, int portToLookFor, String hostType) {
        SearchParameters criteria = new SearchParameters().setType(Type.valueOf(hostType));
        List<Host> hosts = repository.listEntities(criteria);
        for (Host host : hosts) {
            String address = null;
            int port = 0;
            try {
                address = host.getProperty(ConnectionOptions.ADDRESS);
                if (host.hasProperty(ConnectionOptions.PORT)) {
                    port = host.getProperty(ConnectionOptions.PORT);
                }
            } catch (Exception e) {
                // ignore. This happens when looking at a local connection which does not have an address
            }
            if (addressToLookFor.equals(address) && portToLookFor == port) {
                return host;
            }
        }
        return null;
    }

    public Host createJenkinsHost(String address, String jobName) {
        Host host = DescriptorRegistry.getDescriptor(Type.valueOf(HOST_TYPE_OVERTHERE_JENKINS)).newInstance("");
        host.setProperty("address", address);
        // set some dummy or guessed values. Note we just need a Host, not particularly a correctly configured one
//        host.setProperty("username", "jenkinsUser");
//        host.setProperty("password", "jenkinsUserPassword");
        host.setOs(OperatingSystemFamily.UNIX);
        host.setProperty("jobName", jobName);
        host.setId("Infrastructure/" + jobName);
        repository.create(host);
        return host;
    }

    public Host findOrCreateJenkinsHost(String jenkinsUrl, String jobName) {
        Host host = findJenkinsHost(jenkinsUrl);
        if (host != null) {
            return host;
        }
        // no existing Host found, so create a new one
        return createJenkinsHost(jenkinsUrl, jobName);
    }

    public static Host create(String type, String address, String username, String password, String jobName) {
        Host host;
        if ("overthere.LocalHost".equalsIgnoreCase(type)) {
            host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.LocalHost")).newInstance("");
            host.setOs(OperatingSystemFamily.getLocalHostOperatingSystemFamily());
        } else if ("overthere.SshHost".equalsIgnoreCase(type)) {
            host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.SshHost")).newInstance("");
            host.setProperty("address", address);
            host.setProperty("username", username);
            host.setProperty("password", password);
            host.setOs(OperatingSystemFamily.UNIX);
        } else if ("overthere.CifsHost".equalsIgnoreCase(type)) {
            host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.CifsHost")).newInstance("");
            host.setProperty("address", address);
            host.setProperty("username", username);
            host.setProperty("password", password);
            host.setOs(OperatingSystemFamily.WINDOWS);
        } else if ("overthere.JenkinsHost".equalsIgnoreCase(type)) {
            host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.JenkinsHost")).newInstance("");
            host.setProperty("address", address);
            host.setProperty("username", username);
            host.setProperty("password", password);
            host.setProperty("jobName", jobName);
            host.setOs(OperatingSystemFamily.UNIX);
        } else {
            throw new AssertionError(format("Protocol %s is not supported.", type));
        }
        return host;
    }
}
