package com.xebialabs.xltest.repository;

import java.io.*;
import java.security.*;
import java.security.cert.Certificate;
import java.util.Enumeration;
import java.util.PropertyPermission;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.CharStreams;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static java.lang.Thread.currentThread;

/**
 * XL Script executor.
 * <p/>
 * Throws `ScriptExecutionException` if the script should have been executed (for example: a file location was
 * provided).
 * <p/>
 * `FileNotFoundException` is thrown in case a resource can not be found (e.g. if the executor defaults to a
 * CI default script name).
 */
public class ScriptExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(ScriptExecutor.class);

    public static final String ENGINE = "python";
    public static final String SCRIPT_EXTENSION = ".py";
    public static final String SCRIPT_LOCATION_PROPERTY = "scriptLocation";

    private ScriptEngine scriptEngine;
    private AccessControlContext accessControlContext;

    public ScriptExecutor(PermissionCollection permissions) {
        this.scriptEngine = initScriptEngine();
        this.accessControlContext = initAccessControlContext(permissions);
    }

    public void evalScript(final String script, final ScriptContext scriptContext) throws ScriptExecutionException {
        try {
            scriptEngine.eval(script, scriptContext);
        } catch (ScriptException e) {
            throw new ScriptExecutionException("Unable to evaluate script <<<" + script + ">>>", e);
        }
    }

    private ScriptEngine initScriptEngine() {
        ScriptEngineManager manager = new ScriptEngineManager();
        ScriptEngine engine = manager.getEngineByName(ENGINE);
        return engine;
    }

    private AccessControlContext initAccessControlContext(PermissionCollection permissions) {
        ProtectionDomain domain = new ProtectionDomain(new CodeSource(null, (Certificate[]) null), extendsMinimalPermissionsWith(permissions));
        return new AccessControlContext(new ProtectionDomain[]{domain});
    }

    private Permissions extendsMinimalPermissionsWith(PermissionCollection permissions) {
        Permissions minimalPermissions = new Permissions();
        minimalPermissions.add(new PropertyPermission("user.dir", "read"));
        minimalPermissions.add(new RuntimePermission("createClassLoader"));
        minimalPermissions.add(new RuntimePermission("getProtectionDomain"));
        addJythonLibrariesReadPermission(minimalPermissions);

        if (permissions != null) {
            Enumeration<Permission> permissionEnumeration = permissions.elements();
            while (permissionEnumeration.hasMoreElements()) {
                minimalPermissions.add(permissionEnumeration.nextElement());
            }
        }

        return minimalPermissions;
    }

    private void addJythonLibrariesReadPermission(Permissions minimalPermissions) {
        addReadPermissionOnJar("jython-standalone", minimalPermissions);

        // Required for httplib!
        minimalPermissions.add(new PropertyPermission("os.name", "read"));
        minimalPermissions.add(new PropertyPermission("os.arch", "read"));
    }

    private void addReadPermissionOnJar(String jarName, Permissions minimalPermissions) {
        String classPath = System.getProperty("java.class.path");
        String classPathSeparator = System.getProperty("path.separator");
        Pattern pattern = Pattern.compile("([^" + classPathSeparator + "]*" + jarName + "[^" + classPathSeparator + "]*)");
        Matcher matcher = pattern.matcher(classPath);
        while (matcher.find()) {
            String group = matcher.group();
            minimalPermissions.add(new FilePermission(group, "read"));
            LOG.debug("Minimal permission added on: {}", group);
        }
    }

    public void evalScriptedCi(ConfigurationItem ci, ScriptContext scriptContext) throws ScriptExecutionException, FileNotFoundException {
        evalScriptedCi(ci, SCRIPT_LOCATION_PROPERTY, scriptContext);
    }

    public void evalScriptedCi(ConfigurationItem ci, String scriptLocationPropertyName, ScriptContext scriptContext) throws ScriptExecutionException, FileNotFoundException {
        String scriptLocation = ci.getProperty(scriptLocationPropertyName);
        if (scriptLocation == null || scriptLocation.equals("")) {
            String defaultPath = getDefaultPath(ci);
            LOG.info("falling back to default location: " + defaultPath);
            evalScriptFile(defaultPath, scriptContext);
        } else {
            try {
                evalScriptFile(scriptLocation, scriptContext);
            } catch (FileNotFoundException e) {
                throw new ScriptExecutionException("Can not execute script: " + e.getMessage(), e);
            }
        }
    }

    private String getDefaultPath(ConfigurationItem ci) {
        Type type = ci.getType();
        String prefix = type.getPrefix();
        String name = type.getName();

        return prefix + "/" + name + SCRIPT_EXTENSION;
    }

    private String readScript(String path) throws ScriptExecutionException, FileNotFoundException {
        InputStream stream = currentThread().getContextClassLoader().getResourceAsStream(path);
        if (null == stream) {
            throw new FileNotFoundException("Unable to find script '" + path + "' in class path.");
        }

        try {
            return CharStreams.toString(new InputStreamReader(stream, "UTF-8"));
        } catch (IOException e) {
            throw new ScriptExecutionException("Unable to read script file '" + path + "'", e);
        } finally {
            try {
                stream.close();
            } catch (IOException e) {
                LOG.warn("Exception while closing stream for script '{}'", path, e);
            }
        }
    }

    public void evalScriptFile(String path, ScriptContext scriptContext) throws ScriptExecutionException, FileNotFoundException {
        LOG.info("Will execute script {} in context {}", path, scriptContext);
        String script = readScript(path);
        try {
            evalScript(script, scriptContext);
        } catch (Throwable t) {
            throw new ScriptExecutionException("Failed to evaluate script '" + path + "'", t);
        }
        LOG.info("Script {} executed successfully.", path);
    }
}
