# Parse .trx result files.
#
# This module can be used as is. No guarantees can be made on the correctness of the output.
#
#

from xml.etree import ElementTree as ET
from hashlib import md5
from datetime import timedelta
from java.time import OffsetDateTime
from com.xebialabs.xlt.plugin.api.testrun import Event
from org.python.core.util import FileUtil
from com.xebialabs.xlt.plugin.api.resultparser import UnexpectedFormatException


# The .trx file contains an xml namespace which is needed for parsing
ns = {'ms': 'http://microsoft.com/schemas/VisualStudio/TeamTest/2010'}


def extract_last_modified(root):
    timestamp = root.find('ms:Times', ns).attrib['creation']
    return OffsetDateTime.parse(timestamp).toInstant().toEpochMilli()


def extract_run_key(root_nodes):
    """
    Create a unique and deterministic key for the given list of .trx files.

    :param root_nodes: xml root nodes of .trx files
    :return: unique run key
    """
    runkey = ""
    for root in root_nodes:
        runkey += root.attrib['id']

    return md5(runkey).hexdigest()

def extract_hierarchy(test_node):
    """
    Returns an hierarchy path for the test node
    :param test_node: XML node of UnitTest
    :return: the hierarchy path
    """
    hierarchy = test_node.find('ms:TestMethod', ns).attrib['className'].split(",")[0].split('.') + [test_node.attrib['name']]
    return hierarchy

def open_file(file):
    return FileUtil.wrap(file.getInputStream())


def get_root_node(file):
    """
    Return the root XML node of file, no child nodes.

    :param file: `java.io.File` to read.
    :return: a bare ElementTree node with the root node.
    """

    with open_file(file) as f:
        event, rootNode = ET.iterparse(f, events=('start',)).next()
        return rootNode


def throw_if_some_failed(input_files, accepted_files):
    """ Determine if some files in the list input_files are not in the accepted_files set.
        If some files were not accepted then an UnexpectedFormatException is thrown.
    :param input_files: a list of files.
    :param accepted_files: a set of files that was accepted.
    """
    diff = set(input_files).difference(accepted_files)
    if diff:
        msg = "Canceled the import. The following files were not accepted by the test tool: " + ", ".join(map(str, diff))
        raise UnexpectedFormatException(msg, diff)


def validate_files(files):
    """
     Validate that all `files` are .trx files.

     Raises an exception if not all files are valid.

     :param files: a set of files to validate.

     :return: the set of files that was accepted by the tool.
     """
    # Does a basic validation - only checks that all files have a .trx extension
    filtered = []

    for file in files:
        if str(file).endswith("trx"):
            filtered.append(file)
    throw_if_some_failed(files, filtered)
    return filtered


def parse_duration(duration):
    """
    Parses the .trx duration format into milliseconds
    :param duration: duration in the format hh:mm:ss.fraction, for example: 00:00:03.0459057
    :return: duration value in milliseconds
    """
    time = duration.split(":")
    time_delta = timedelta(hours=float(time[0]), minutes=float(time[1]), seconds=float(time[2]))
    return time_delta.microseconds / 1000


def extract_duration(test_case_node):
    """
    Extract the duration of the test case in milliseconds from the test case node, as provided by `iterate_test_cases`.

    :param test_case_node: the test case XML node to extract the duration from.

    :return: the duration.
    """
    try:
        duration = test_case_node.attrib["duration"]
        return parse_duration(duration)
    except KeyError:
        return 0


def extract_result(test_case_node):
    """
    Extract the test result (a string "Passed or "Failed") from the test case node, as provided by `iterate_test_cases`.

    :param test_case_node: the test case XML node to extract the result from

    :return: the result of the testcase.
    """
    outcome = test_case_node.attrib["outcome"]
    if outcome == 'Passed':
       return "PASSED"
    # MSTest reports a timeout explicitly, while VSTest reports it as Failed
    elif outcome =='Failed' or outcome == 'Timeout':
        return "FAILED"
    else:
       return "OTHER"

def extract_failure_reason(test_case_node):
    """
    Extract the failure reason from the test case node if the test failed.

    :param test_case_node: a test case XML node, as provided by `iterate_test_cases`.

    :return: the failure reason, or None if the there was no failure.
    """
    try:
       return test_case_node.find('.//{%s}Message' % ns['ms']).text.strip()
    except AttributeError:
       return None


def convert_test_result_to_event(test_case_node, storage, hierarchy):
    """ Extract the last modified time from the .trx file.

    :param file: the .trx file.
    :param test_case_node: the test case result XML node.
    :param storage: the DLL file name for the test case node.
    :param hierarchy: list with the test case hierarchy.


    :return: a map representing the event to generate
    """
    test_case_duration = extract_duration(test_case_node)
    event_map = {
        Event.TYPE: Event.TYPE_FUNCTIONAL_RESULT,
        Event.DURATION: test_case_duration,
        Event.HIERARCHY: hierarchy,
        Event.RESULT: extract_result(test_case_node),
        Event.FIRST_ERROR: extract_failure_reason(test_case_node),
        'storage': storage
    }
    return event_map, test_case_duration


def iterate_test_cases(file):
    """
    Iterate all test cases found in `file`.

    :param file: the .trx file to extract the test cases from.

    :return: a list/iterator of tuples (result node, test hierarchy path).
    """
    with open_file(file) as f:
        tests = []
        results = []
        et_iter = iter(ET.iterparse(f, events=('start', 'end',)))
        event, root_node = et_iter.next()
        for event, elem in et_iter:
            if event == 'end':
                if elem.tag == '{%s}UnitTest' % ns['ms']:
                    tests.append(elem)
                elif elem.tag == '{%s}UnitTestResult' % ns['ms']:
                    results.append(elem)
            root_node.clear()

        # the .trx format has different elements for tests and execution results, with a shared execution id
        # create 2 dictionaries which map the execution id to each test and result
        tests_by_id = { test.find('ms:Execution', ns).attrib['id']: test for test in tests if test.find('ms:Execution', ns) is not None }
        results_by_id = { result.attrib['executionId']: result for result in results }

        return map(lambda (id, result): (result, tests_by_id[id].get('storage'), extract_hierarchy(tests_by_id[id])),
                   results_by_id.iteritems())


def parse_test_results(files, runkey, last_modified):
    """ Parse .trx results from a set of files.

    :param files: the .trx files that should be imported.
    :param runkey: a unique key for this test run.
    :param last_modified: the most recent last modified time of the files.

    :return: a list of events representing the test run
    """
    events = []
    test_run_duration = 0

    # Add import started event always as first
    import_started_event = Event.createImportStartedEvent(last_modified)
    import_started_event.update(Event.RUN_KEY, runkey)
    events.insert(0, import_started_event)

    for file in files:
        for test_case_node, storage, hierarchy in iterate_test_cases(file):
            event_map, test_case_duration = convert_test_result_to_event(test_case_node, storage, hierarchy)
            test_run_duration += test_case_duration

            events.append(Event(event_map))

    events.append(Event.createImportFinishedEvent(test_run_duration))

    return events



# Start parsing run

validate_files(files)

# sort files to ensure the same order when creating the run key
root_nodes = map(get_root_node, sorted(files, key=lambda file: file.getName()))
run_key = extract_run_key(root_nodes)
last_modified = max(map(extract_last_modified, root_nodes))

if not test_run_historian.isKnownKey(run_key):
    events = parse_test_results(files, run_key, last_modified)
else:
    events = []

# Result holder should contain a list of test runs. A test run is a list of events.
result_holder.result = [events] if events else []
