"""
Helper functions for building functional test result parsers.

The main entry point for a xUnit type test result parser is `parse_junit_test_results`.
"""

from xml.etree import ElementTree as ET
from datetime import datetime
from copy import deepcopy
from org.python.core.util import FileUtil

from com.xebialabs.xlt.plugin.api.testrun import Event
from com.xebialabs.xlt.plugin.api.resultparser import UnexpectedFormatException
from com.xebialabs.xlt.plugin.api.resultparser import MalformedInputException


time_format = '%Y-%m-%dT%H:%M:%S'

def open_file(file):
    """
    Make an overthere file usable as a python file.
    :param file: o  verthere file
    :return: a python file object
    """
    return FileUtil.wrap(file.getInputStream())

def throw_if_some_failed(input_files, accepted_files):
    """ Determine if some files in the list input_files are not in the accepted_files set. If some files were not accepted then an UnexpectedFormatException is thrown.
    :param input_files: a list of files.
    :param accepted_files: a set of files that was accepted.
    """
    diff = set(input_files).difference(accepted_files)
    if diff:
        msg = "Canceled the import. The following files were not accepted by the test tool: " + ", ".join(map(str, diff))
        raise UnexpectedFormatException(msg, diff)


def get_root_node(file):
    """
    Return the root XML node of file, no child nodes.

    :param file: `java.io.File` to read.
    :return: a bare ElementTree node with the root node.
    """

    with open_file(file) as f:
        event, rootNode = ET.iterparse(f, events=('start',)).next()
        return rootNode


def validate_timestamp(root):
    """
    Validate that a valid timestamp attribute exists in the given xml root node

    :param root: xml root node
    :return: True if a valid timestamp exists, False otherwise
    """
    timestamp = root.attrib["timestamp"]
    try:
        datetime.strptime(timestamp, time_format)
        return True
    except ValueError:
        return False

def validate_files(files, validate_timestamp=validate_timestamp):
    """
    Validate that all `files` are xUnit files.

    Raises an exception if not all files are valid.

    :param files: a set of files to validate.

    :return: the set of files that was accepted by the tool.
    """
    filtered = []
    for file in files:
        if str(file).endswith("xml"):
            try:
                root = get_root_node(file)
                # Use endswith instead of equals to handle xml namespaces
                if root.tag.endswith("testsuite") and validate_timestamp(root):
                  filtered.append(file)
            except:
                pass
    throw_if_some_failed(files, filtered)
    return filtered


def iterate_test_cases(file):
    """
    Iterate all test cases found in `file`.

    :param file: the XUnit file to extract the test cases from.

    :return: a list/iterator of tuples (test case node, test hierarchy path).
    """
    with open_file(file) as f:
        et_iter = iter(ET.iterparse(f, events=('start', 'end',)))
        event, root_node = et_iter.next()
        path = root_node.attrib['name'].split('.')
        for event, elem in et_iter:
            if event == 'end' and elem.tag == 'testcase':
                yield (elem, path + [elem.attrib['name']])
                elem.clear()


def extract_duration(test_case_node):
    """
    Extract the duration of the test case in milliseconds from the test case node, as provided by `iterate_test_cases`.

    :param test_case_node: the test case XML node to extract the duration from.

    :return: the duration.
    """
    time = test_case_node.attrib["time"]
    # Numbers are expected to be in american format, with dots as the decimal separator
    # and optionally commas as the thousands separator, e.g. 1,234.54, according to XML specification
    if time is not None:
        time = time.replace(",","")
    return int(float(time) * 1000)


def extract_result(test_case_node):
    """
    Extract the test result (a string "PASSED or "FAILED") from the test case node, as provided by `iterate_test_cases`.

    If the result is neither of the two (ie, skipped, or ignored, etc). Then the result should be OTHER.

    :param test_case_node: the test case XML node to extract the result from

    :return: the result of the testcase.
    """
    failure = test_case_node.find("failure")
    error = test_case_node.find("error")
    skipped = test_case_node.find("skipped")
    if failure is not None or error is not None:
        return "FAILED"
    elif skipped is not None:
        return "OTHER"
    else:
        return "PASSED"


def extract_failure_reason(test_case_node):
    """
    Extract the failure reason from the test case node if the test failed.

    :param test_case_node: a test case XML node, as provided by `iterate_test_cases`.

    :return: the failure reason, or None if the there was no failure.
    """
    if extract_result(test_case_node) == "FAILED":
        failure = test_case_node.find("failure")
        error = test_case_node.find("error")
        if failure is not None:
            return failure.attrib.get("message", "No failure reason").strip()
        elif error is not None:
            return error.attrib.get("message", "No failure reason").strip()
        else:
            raise MalformedInputException("Expected to have found a failure reason!")
    else:
        return None


def extract_custom_properties(test_case_node, file):
    """ Extract custom properties from the `test_case_node`. Enhance this function to provide extra information for reports. Extra properties must not
    start with a '@' nor with the prefix 'ci'. It is recommended to choose a prefix for your organization and prefix all properties with that.

    :param test_case_node: the test case XML node to extract the custom properties from.
    :param file: the file the node was extracted from.

    :return: a map of extra properties.
    """
    return {}

def extract_last_modified(file):
    """ Extract the last modified time from the xUnit file.

    :param file: the xUnit file for which the last modified time has to be determined.

    :return: the last modified time as milliseconds since epoch.
    """
    root = get_root_node(file)
    timestamp = root.attrib["timestamp"]
    return int(((datetime.strptime(timestamp, time_format) - datetime(1970, 1, 1)).total_seconds() * 1000))


def convert_test_result_to_event(file, test_case_node, hierarchy, extract_result, extract_duration,
                                 extract_failure_reason, extract_custom_properties):
    """ Extract the last modified time from the xUnit file.

    :param file: the xUnit file.
    :param test_case_node: the test case XML node.
    :param hierarchy: list with the test case hierarchy.
    :param extract_result: function used to extract the result of a test case.
    :param extract_duration: function used to extract the duration of a test case.
    :param extract_failure_reason: function used to extract the failure reasone of a test case.
    :param extract_custom_properties: function used to enhance the generated events with extra properties.

    :return: a map representing the event to generate
    """
    test_case_duration = extract_duration(test_case_node)
    event_map = {
        Event.TYPE: Event.TYPE_FUNCTIONAL_RESULT,
        Event.DURATION: test_case_duration,
        Event.HIERARCHY: hierarchy,
        Event.RESULT: extract_result(test_case_node),
        Event.FIRST_ERROR: extract_failure_reason(test_case_node)
    }
    event_map.update(extract_custom_properties(test_case_node, file))
    return event_map, test_case_duration


# TODO: parse -> find last modified
def parse_last_modified(files, extract_last_modified=extract_last_modified):
    """ Determine the most recent last modified time of a set of xUnit files.

    :param files: the xUnit files for which the last modified time has to be determined.
    :param extract_last_modified: function used to extract the last modified time from a file.

    :return: the last modified time as milliseconds since epoch.
    """
    newest_timestamp = 0
    for file in files:
        file_timestamp = extract_last_modified(file)
        newest_timestamp = newest_timestamp if newest_timestamp > file_timestamp else file_timestamp
    return newest_timestamp


def merge_results(*results):
    """
    Merges a group of results into one single result.
    Used when merging events which share the same hierarchy.
    The rules for merging the results are:
    1) If any result is failed, then the merged result is failed.
    2) If there are no failed results, then if any result is "other", then the merged result is other.
    3) If there are no failed or other results, then the merged result is passed

    :param results: a list of result strings which is either FAILED, OTHER or PASSED
    :return: a single result which describes the outcome
    """
    if 'FAILED' in results:
        return 'FAILED'
    elif 'OTHER' in results:
        return 'OTHER'
    else:
        return 'PASSED'

def merge_first_error(e1, e2):
    """
    Merges the error message from the 2 passed events.
    Used when merging events which share the same hierarchy.
    The rules for merging the errors for 2 events are:
    1) If the first event has a non empty error value, return it
    2) If the second event has a non empty error value, return it
    3) If there are no errors, return None

    :param e1: first event map
    :param e2: second event map
    :return: a single error message
    """
    if e1.has_key(Event.FIRST_ERROR) and e1[Event.FIRST_ERROR]:
        return e1[Event.FIRST_ERROR]
    return e2.get(Event.FIRST_ERROR)

def parse_junit_test_results(files, last_modified,
                             iterate_test_cases=iterate_test_cases,
                             extract_duration=extract_duration,
                             extract_result=extract_result,
                             extract_failure_reason=extract_failure_reason,
                             extract_custom_properties=extract_custom_properties):
    """ Parse xUnit results from a set of files using various functions that can be overridden.

    :param files: the xUnit files that should be imported.
    :param last_modified: the most recent last modified time of the files.
    :param iterate_test_cases: function used to extract the test cases of a xUnit file.
    :param extract_duration: function used to extract the duration of a test case.
    :param extract_result: function used to extract the result of a test case.
    :param extract_failure_reason: function used to extract the failure of a test case.
    :param extract_custom_properties: function used to extract custom properties.

    :return: a list of events representing the test run
    """
    events = []
    test_run_duration = 0
    events_by_hierarchy = {}

    # Add import started event always as first
    import_started_event = Event.createImportStartedEvent(last_modified)
    import_started_event.update(Event.RUN_KEY, str(last_modified))
    events.insert(0, import_started_event)

    for file in files:
        for test_case_node, hierarchy in iterate_test_cases(file):
            current_event, test_case_duration = convert_test_result_to_event(file, test_case_node, hierarchy,
                                                                         extract_result, extract_duration,
                                                                         extract_failure_reason,
                                                                         extract_custom_properties)
            test_run_duration += test_case_duration

            hierarchy_key = tuple(hierarchy)
            # check if we already have an event with this hierarchy, and merge results if needed
            try:
                event_with_same_hierarchy = events_by_hierarchy[hierarchy_key]
            except KeyError:
                events_by_hierarchy[hierarchy_key] = current_event
                continue


            if not event_with_same_hierarchy.has_key(Event.MERGED_EVENTS):
                # performing a copy in order to prevent a circular reference
                event_with_same_hierarchy[Event.MERGED_EVENTS] = [deepcopy(event_with_same_hierarchy)]

            event_with_same_hierarchy[Event.MERGED_EVENTS].append(current_event)
            event_with_same_hierarchy[Event.DURATION] += current_event[Event.DURATION]
            event_with_same_hierarchy[Event.RESULT] = merge_results(event_with_same_hierarchy[Event.RESULT], current_event[Event.RESULT])
            event_with_same_hierarchy[Event.FIRST_ERROR] = merge_first_error(event_with_same_hierarchy, current_event)

    events.extend(Event(ev) for ev in events_by_hierarchy.itervalues())
    events.append(Event.createImportFinishedEvent(test_run_duration))

    return events

